package ru.yandex.direct.api.v5.entity.retargetinglists.validation;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.retargetinglists.AddRequest;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListAddItem;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListRuleItem;
import com.yandex.direct.api.v5.retargetinglists.RetargetingListUpdateItem;
import com.yandex.direct.api.v5.retargetinglists.UpdateRequest;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.entity.retargetinglists.RetargetingListsTranslations;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.retargetinglists.Constants.MAX_RET_CONDITIONS_PER_REQUEST;
import static ru.yandex.direct.api.v5.entity.retargetinglists.converter.GoalInterestsTypeConverter.getInterestType;
import static ru.yandex.direct.api.v5.entity.retargetinglists.converter.GoalInterestsTypeConverter.getInternalId;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.core.entity.retargeting.model.GoalType.INTERESTS;

@Component
@ParametersAreNonnullByDefault
public class AddUpdateRequestValidator {

    private static final DefectType INTERESTS_MUST_BE_SAME_TERM =
            new DefectType(8000, RetargetingListsTranslations.INSTANCE.interestsMustHaveSameTerm());

    private static final String RETARGETING_LIST_NAME = AddRequest.PropInfo.RETARGETING_LISTS.schemaName.getLocalPart();
    private static final String RULES_NAME = RetargetingListAddItem.PropInfo.RULES.schemaName.getLocalPart();

    public static ValidationResult<AddRequest, DefectType> validateAddRequest(AddRequest externalRequest) {
        ItemValidationBuilder<AddRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getRetargetingLists(), RETARGETING_LIST_NAME)
                .check(maxListSize(MAX_RET_CONDITIONS_PER_REQUEST),
                        maxElementsPerRequest(MAX_RET_CONDITIONS_PER_REQUEST));

        vb.list(externalRequest.getRetargetingLists(), RETARGETING_LIST_NAME)
                .checkEachBy(validateAddItem());

        return vb.getResult();
    }

    private static Validator<RetargetingListAddItem, DefectType> validateAddItem() {
        return listAddItem -> {
            ItemValidationBuilder<RetargetingListAddItem, DefectType> v = ItemValidationBuilder.of(listAddItem);
            v.list(listAddItem.getRules(), RULES_NAME)
                    .checkEach(interestsMustBeSameTerm());

            return v.getResult();
        };
    }

    public static ValidationResult<UpdateRequest, DefectType> validateUpdateRequest(UpdateRequest externalRequest) {
        ItemValidationBuilder<UpdateRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);
        vb.item(externalRequest.getRetargetingLists(), RETARGETING_LIST_NAME)
                .check(maxListSize(MAX_RET_CONDITIONS_PER_REQUEST),
                        maxElementsPerRequest(MAX_RET_CONDITIONS_PER_REQUEST));

        vb.list(externalRequest.getRetargetingLists(), RETARGETING_LIST_NAME)
                .checkEachBy(validateUpdateItem());

        return vb.getResult();
    }

    private static Validator<RetargetingListUpdateItem, DefectType> validateUpdateItem() {
        return listAddItem -> {
            ItemValidationBuilder<RetargetingListUpdateItem, DefectType> v = ItemValidationBuilder.of(listAddItem);
            v.list(listAddItem.getRules(), RULES_NAME)
                    .checkEach(interestsMustBeSameTerm());

            return v.getResult();
        };
    }

    // Тип интереса вычисляем по первой цели.
    // И длительность всех интересов (цели с типом INTERESTS) должна совпадать
    private static Constraint<RetargetingListRuleItem, DefectType> interestsMustBeSameTerm() {
        return Constraint.fromPredicate(
                rule -> rule.getArguments().stream()
                        .filter(arg -> Goal.computeType(getInternalId(arg.getExternalId())) == INTERESTS)
                        .map(argument -> getInterestType(argument.getExternalId()))
                        .distinct()
                        .count() <= 1,
                INTERESTS_MUST_BE_SAME_TERM);
    }
}
