package ru.yandex.direct.api.v5.entity.sitelinks.delegate;

import java.util.List;

import javax.annotation.Nullable;

import com.yandex.direct.api.v5.sitelinks.AddRequest;
import com.yandex.direct.api.v5.sitelinks.AddResponse;
import com.yandex.direct.api.v5.sitelinks.SitelinkAddItem;
import com.yandex.direct.api.v5.sitelinks.SitelinksSetAddItem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.sitelinks.SitelinksEndpoint;
import ru.yandex.direct.api.v5.result.ApiMassResult;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.sitelink.model.Sitelink;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.api.v5.entity.sitelinks.Constants.MAX_SITELINKS_PER_ADD;
import static ru.yandex.direct.api.v5.validation.DefectTypes.absent;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.minListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


/**
 * Добавить наборы быстрых ссылок
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/sitelinks/get-docpage/
 */
@Component
public class AddSitelinksDelegate extends
        BaseApiServiceDelegate<AddRequest, AddResponse, List<SitelinkSet>, ApiResult<Long>> {
    private final SitelinkSetService service;
    private final ResultConverter resultConverter;

    @Autowired
    public AddSitelinksDelegate(
            SitelinkSetService service,
            ApiAuthenticationSource auth,
            ResultConverter resultConverter) {
        super(SitelinksEndpoint.PATH_CONVERTER, auth);

        this.service = service;
        this.resultConverter = resultConverter;
    }

    @Override
    public List<SitelinkSet> convertRequest(AddRequest externalRequest) {
        return mapList(externalRequest.getSitelinksSets(), AddSitelinksDelegate::convertSet);
    }

    @Override
    public ApiMassResult<Long> processRequest(List<SitelinkSet> request) {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        ValidationResult<List<SitelinkSet>, DefectType> validation = validate(request);
        if (validation.hasAnyErrors()) {
            return ApiMassResult.brokenMassAction(emptyList(), validation);
        }
        return resultConverter.toApiMassResult(service.addSitelinkSetsPartial(clientId, request));
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new AddResponse()
                .withAddResults(resultConverter.toActionResults(result, apiPathConverter));
    }

    private static SitelinkSet convertSet(SitelinksSetAddItem set) {
        return new SitelinkSet()
                .withSitelinks(mapList(set.getSitelinks(), AddSitelinksDelegate::convertLink));
    }

    private static Sitelink convertLink(SitelinkAddItem link) {
        return new Sitelink()
                .withTurboLandingId(link.getTurboPageId() != null ? Long.valueOf(link.getTurboPageId()) : null)
                .withTitle(trim(link.getTitle()))
                .withDescription(trim(link.getDescription()))
                .withHref(trim(link.getHref()));
    }

    private static ValidationResult<List<SitelinkSet>, DefectType> validate(List<SitelinkSet> data) {
        return ListValidationBuilder.<SitelinkSet, DefectType>of(data)
                .check(notNull())
                .check(minListSize(1), absent())
                .check(maxListSize(MAX_SITELINKS_PER_ADD), maxElementsPerRequest(MAX_SITELINKS_PER_ADD))
                .getResult();
    }

    private static String trim(@Nullable String string) {
        return string != null ? string.trim() : null;
    }
}
