package ru.yandex.direct.api.v5.entity.sitelinks.delegate;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.sitelinks.DeleteRequest;
import com.yandex.direct.api.v5.sitelinks.DeleteResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.sitelinks.SitelinksEndpoint;
import ru.yandex.direct.api.v5.entity.sitelinks.converter.DeleteRequestConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.api.v5.validation.DefectTypes;
import ru.yandex.direct.api.v5.validation.constraints.Constraints;
import ru.yandex.direct.core.entity.sitelink.service.SitelinkSetService;
import ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkDefectTranslations;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.i18n.types.Identity;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.sitelinks.Constants.MAX_IDS_PER_DELETE;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;

/**
 * Удалить наборы быстрых ссылок
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/sitelinks/delete-docpage/
 */
@Component
@ParametersAreNonnullByDefault
public class DeleteSitelinksDelegate extends BaseApiServiceDelegate<DeleteRequest, DeleteResponse, List<Long>, ApiResult<Long>> {
    private static final DefectType DELETE_MAX_IDS_DETAILED = DefectTypes.maxIdsInSelection().withDetailedMessage(
            SitelinkDefectTranslations.INSTANCE.deleteMaxIdsDetailed(new Identity(MAX_IDS_PER_DELETE)));

    private final DeleteRequestConverter requestConverter;
    private final ResultConverter resultConverter;
    private final SitelinkSetService service;

    @Autowired
    public DeleteSitelinksDelegate(ApiAuthenticationSource auth,
                                   DeleteRequestConverter requestConverter,
                                   ResultConverter resultConverter,
                                   SitelinkSetService service) {
        super(SitelinksEndpoint.PATH_CONVERTER, auth);
        this.requestConverter = requestConverter;
        this.resultConverter = resultConverter;
        this.service = service;
    }

    @Override
    public ValidationResult<DeleteRequest, DefectType> validateRequest(DeleteRequest request) {
        ItemValidationBuilder<DeleteRequest, DefectType> vb = ItemValidationBuilder.of(request);
        vb.item(request.getSelectionCriteria(), "SelectionCriteria")
                .item(request.getSelectionCriteria().getIds(), "Ids")
                .check(Constraints.maxListSize(MAX_IDS_PER_DELETE), DELETE_MAX_IDS_DETAILED)
                .check(eachNotNull());
        return vb.getResult();
    }

    @Override
    public List<Long> convertRequest(DeleteRequest externalRequest) {
        return requestConverter.convert(externalRequest);
    }

    @Override
    public ApiResult<List<ApiResult<Long>>> processRequest(List<Long> sitelinkSetIds) {
        return resultConverter.toApiMassResult(service.deleteSiteLinkSets(getClientId(), sitelinkSetIds));
    }

    private ClientId getClientId() {
        return auth.getChiefSubclient().getClientId();
    }

    @Override
    public DeleteResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new DeleteResponse()
                .withDeleteResults(resultConverter.toActionResults(result, apiPathConverter));
    }

}
