package ru.yandex.direct.api.v5.entity.smartadtargets.converter;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.general.ConditionTypeEnum;
import com.yandex.direct.api.v5.general.PriorityEnum;
import com.yandex.direct.api.v5.general.StateEnum;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.smartadtargets.ConditionsArray;
import com.yandex.direct.api.v5.smartadtargets.ConditionsItem;
import com.yandex.direct.api.v5.smartadtargets.GetResponse;
import com.yandex.direct.api.v5.smartadtargets.SmartAdTargetFieldEnum;
import com.yandex.direct.api.v5.smartadtargets.SmartAdTargetGetItem;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ConverterUtils;
import ru.yandex.direct.api.v5.common.GeneralUtil;
import ru.yandex.direct.api.v5.entity.smartadtargets.container.GetTargetContainer;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableMap;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToMicros;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.API_OPERATOR_BY_CORE_OPERATOR;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.AUDIENCE_BY_FUNNEL;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.FACTORY;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.IS_AVAILABLE_CONDITION;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.IS_NOT_AVAILABLE_CONDITION;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.PRICE_OFF;


@ParametersAreNonnullByDefault
@Service
public class GetResponseConverterService {

    private static final Map<SmartAdTargetFieldEnum, ItemFieldApplier> APPLIER_BY_FIELD = getApplierByField();

    private final CampaignService campaignService;

    @Autowired
    public GetResponseConverterService(CampaignService campaignService) {

        this.campaignService = campaignService;
    }

    private static Map<SmartAdTargetFieldEnum, ItemFieldApplier> getApplierByField() {
        EnumMap<SmartAdTargetFieldEnum, ItemFieldApplier> map =
                new EnumMap<>(SmartAdTargetFieldEnum.class);
        map.put(SmartAdTargetFieldEnum.AD_GROUP_ID, GetResponseConverterService::applyAdGroup);
        map.put(SmartAdTargetFieldEnum.AUDIENCE, GetResponseConverterService::applyAudience);
        map.put(SmartAdTargetFieldEnum.AVAILABLE_ITEMS_ONLY, GetResponseConverterService::applyAvailableItemsOnly);
        map.put(SmartAdTargetFieldEnum.AVERAGE_CPC, GetResponseConverterService::applyAverageCpc);
        map.put(SmartAdTargetFieldEnum.AVERAGE_CPA, GetResponseConverterService::applyAverageCpa);
        map.put(SmartAdTargetFieldEnum.CAMPAIGN_ID, GetResponseConverterService::applyCampaignId);
        map.put(SmartAdTargetFieldEnum.CONDITION_TYPE, GetResponseConverterService::applyConditionType);
        map.put(SmartAdTargetFieldEnum.CONDITIONS, GetResponseConverterService::applyConditions);
        map.put(SmartAdTargetFieldEnum.ID, GetResponseConverterService::applyId);
        map.put(SmartAdTargetFieldEnum.NAME, GetResponseConverterService::applyName);
        map.put(SmartAdTargetFieldEnum.STATE, GetResponseConverterService::applyState);
        map.put(SmartAdTargetFieldEnum.STRATEGY_PRIORITY, GetResponseConverterService::applyStrategyPriority);
        return unmodifiableMap(map);
    }

    private static void applyAdGroup(SmartAdTargetGetItem item, GetTargetContainer container) {
        item.withAdGroupId(container.getFilter().getPid());
    }

    private static void applyAudience(SmartAdTargetGetItem item, GetTargetContainer container) {
        TargetFunnel targetFunnel = container.getFilter().getTargetFunnel();
        item.setAudience(AUDIENCE_BY_FUNNEL.get(targetFunnel));
    }

    private static void applyAvailableItemsOnly(SmartAdTargetGetItem item, GetTargetContainer container) {
        boolean hasAvailableCondition = container.getFilter().getConditions().stream().anyMatch(IS_AVAILABLE_CONDITION);
        YesNoEnum val = GeneralUtil.yesNoFromBool(hasAvailableCondition);
        JAXBElement<YesNoEnum> availableItemsOnly = FACTORY.createSmartAdTargetGetItemAvailableItemsOnly(val);
        item.setAvailableItemsOnly(availableItemsOnly);
    }

    private static void applyAverageCpc(SmartAdTargetGetItem item, GetTargetContainer container) {
        Long cpcMicroPrice = convertToMicros(container.getFilter().getPriceCpc());
        JAXBElement<Long> averageCpc = FACTORY.createSmartAdTargetGetItemAverageCpc(cpcMicroPrice);
        averageCpc.setNil(Objects.equals(cpcMicroPrice, PRICE_OFF));
        item.setAverageCpc(averageCpc);
    }

    private static void applyAverageCpa(SmartAdTargetGetItem item, GetTargetContainer container) {
        Long cpaMicroPrice = convertToMicros(container.getFilter().getPriceCpa());
        JAXBElement<Long> averageCpa = FACTORY.createSmartAdTargetGetItemAverageCpa(cpaMicroPrice);
        averageCpa.setNil(Objects.equals(cpaMicroPrice, PRICE_OFF));
        item.setAverageCpa(averageCpa);
    }

    private static void applyCampaignId(SmartAdTargetGetItem item, GetTargetContainer container) {
        item.setCampaignId(container.getCampaignId());
    }

    private static void applyConditionType(SmartAdTargetGetItem item, GetTargetContainer container) {
        PerformanceFilterTab tab = container.getFilter().getTab();
        ConditionTypeEnum conditionType =
                tab == PerformanceFilterTab.ALL_PRODUCTS ? ConditionTypeEnum.ITEMS_ALL : ConditionTypeEnum.ITEMS_SUBSET;
        item.setConditionType(conditionType);
    }

    private static void applyConditions(SmartAdTargetGetItem item, GetTargetContainer container) {
        List<PerformanceFilterCondition> conditions = container.getFilter().getConditions();
        List<ConditionsItem> conditionsItems = StreamEx.of(conditions)
                .filter(IS_NOT_AVAILABLE_CONDITION)
                .map(GetResponseConverterService::getConditionsItem)
                .toList();
        ConditionsArray conditionsArray = Optional.of(conditionsItems)
                .filter(CollectionUtils::isNotEmpty)
                .map(ci -> new ConditionsArray().withItems(ci))
                .orElse(null);
        JAXBElement<ConditionsArray> itemConditions = FACTORY.createSmartAdTargetGetItemConditions(conditionsArray);
        item.setConditions(itemConditions);
    }

    private static void applyId(SmartAdTargetGetItem item, GetTargetContainer container) {
        item.setId(container.getFilter().getId());
    }

    private static void applyName(SmartAdTargetGetItem item, GetTargetContainer container) {
        item.setName(container.getFilter().getName());
    }

    private static void applyState(SmartAdTargetGetItem item, GetTargetContainer container) {
        PerformanceFilter filter = container.getFilter();
        if (filter.getIsDeleted()) {
            item.setState(StateEnum.DELETED);
            return;
        }
        if (filter.getIsSuspended()) {
            item.setState(StateEnum.SUSPENDED);
            return;
        }
        item.setState(filter.getStatusBsSynced() == StatusBsSynced.YES ? StateEnum.ON : StateEnum.OFF);
    }

    private static void applyStrategyPriority(SmartAdTargetGetItem item, GetTargetContainer container) {
        Integer autobudgetPriority = container.getFilter().getAutobudgetPriority();
        PriorityEnum val = ConverterUtils.convertStrategyPriority(autobudgetPriority);
        JAXBElement<PriorityEnum> strategyPriority = FACTORY.createSmartAdTargetGetItemStrategyPriority(val);
        item.setStrategyPriority(strategyPriority);
    }

    private static SmartAdTargetGetItem convert(GetTargetContainer container,
                                                Set<SmartAdTargetFieldEnum> requestedFields) {
        SmartAdTargetGetItem item = new SmartAdTargetGetItem();
        for (var requestedField : requestedFields) {
            ItemFieldApplier applier = APPLIER_BY_FIELD.get(requestedField);
            applier.apply(item, container);
        }
        return item;
    }

    private static ConditionsItem getConditionsItem(PerformanceFilterCondition perfFilterCond) {
        return new ConditionsItem()
                .withOperand(perfFilterCond.getFieldName())
                .withOperator(API_OPERATOR_BY_CORE_OPERATOR.get(perfFilterCond.getOperator()))
                .withArguments(getArguments(perfFilterCond.getStringValue()));
    }

    private static String[] getArguments(String stringValue) {
        if (stringValue.length() > 2 && stringValue.startsWith("[\"")) {
            stringValue = stringValue.substring(2, stringValue.length() - 2);
        }
        return stringValue.split("\",\"");
    }

    public GetResponse getGetResponse(ClientId clientId, List<PerformanceFilter> filters,
                                      Set<SmartAdTargetFieldEnum> requestedFields,
                                      @Nullable Long limitedBy) {
        List<GetTargetContainer> containers = getContainers(clientId, filters, requestedFields);
        List<SmartAdTargetGetItem> items =
                containers.stream()
                        .map(c -> convert(c, requestedFields))
                        .collect(toList());
        return new GetResponse()
                .withSmartAdTargets(items)
                .withLimitedBy(limitedBy);
    }

    private List<GetTargetContainer> getContainers(ClientId clientId, List<PerformanceFilter> filters,
                                                   Set<SmartAdTargetFieldEnum> requestedFields) {
        Map<Long, Long> campaignIdByFilterId;
        if (requestedFields.contains(SmartAdTargetFieldEnum.CAMPAIGN_ID)) {
            campaignIdByFilterId = campaignService.getCampaignIdByPerfFilterId(clientId, filters);
        } else {
            campaignIdByFilterId = emptyMap();
        }
        return filters.stream()
                .map(f -> new GetTargetContainer(f, campaignIdByFilterId.get(f.getId())))
                .collect(toList());
    }

}
