package ru.yandex.direct.api.v5.entity.smartadtargets.converter;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.validation.constraints.NotNull;
import javax.xml.bind.JAXBElement;

import com.google.common.collect.ImmutableList;
import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.smartadtargets.ConditionsArray;
import com.yandex.direct.api.v5.smartadtargets.ConditionsItem;
import com.yandex.direct.api.v5.smartadtargets.SetBidsItem;
import com.yandex.direct.api.v5.smartadtargets.SetBidsRequest;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertStrategyPriority;
import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.core.entity.performancefilter.model.Operator.EQUALS;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class RequestConverter {

    @SuppressWarnings("rawtypes")
    private static List<PerformanceFilterCondition> getPerformanceFilterConditionsInternal(
            FilterSchema filterSchema,
            @Nullable YesNoEnum availableItemsOnly,
            List<ConditionsItem> items) {
        ImmutableList.Builder<PerformanceFilterCondition> conditionsBuilder = new ImmutableList.Builder<>();
        if (Objects.equals(availableItemsOnly, YesNoEnum.YES)) {
            conditionsBuilder
                    .add(new PerformanceFilterCondition("available", EQUALS, "true"));
        }
        List<PerformanceFilterCondition> conditions = conditionsBuilder
                .addAll(mapList(items, CommonConverters::convertCondition))
                .build();
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions);
        return conditions;
    }

    @SuppressWarnings("rawtypes")
    static List<PerformanceFilterCondition> getPerformanceFilterConditionsForUpdate(
            FilterSchema filterSchema,
            @Nullable YesNoEnum availableItemsOnly,
            @NotNull JAXBElement<ConditionsArray> conditionItems) {
        List<ConditionsItem> items = conditionItems.isNil() ? List.of() : conditionItems.getValue().getItems();
        return getPerformanceFilterConditionsInternal(filterSchema, availableItemsOnly, items);
    }

    @SuppressWarnings("rawtypes")
    static List<PerformanceFilterCondition> getPerformanceFilterConditions(
            FilterSchema filterSchema,
            @Nullable YesNoEnum availableItemsOnly,
            @Nullable ConditionsArray conditionItems) {
        List<ConditionsItem> items = nvl(ifNotNull(conditionItems, ConditionsArray::getItems), List.of());
        return getPerformanceFilterConditionsInternal(filterSchema, availableItemsOnly, items);
    }

    /**
     * Метод повторяеет бизнес-логику конвертера из
     * {@link ru.yandex.direct.internaltools.tools.performance.PerfFilterUpdateTool#getUpdatedTab
     * PerfFilterUpdateTool#getUpdatedTab}
     */
    @SuppressWarnings("JavadocReference")
    static PerformanceFilterTab getTab(@Nullable List<PerformanceFilterCondition> conditions) {
        if (conditions == null) {
            return null;
        }
        if (conditions.isEmpty()) {
            return PerformanceFilterTab.ALL_PRODUCTS;
        }
        if (PerformanceFilterConditionDBFormatParser.isSupportedByTreeTab(conditions)) {
            return PerformanceFilterTab.TREE;
        }
        return PerformanceFilterTab.CONDITION;
    }

    public List<SetBidItem> convertSetBidsRequest(SetBidsRequest setBidsRequest) {
        return mapList(setBidsRequest.getBids(), this::convertSetBidsItem);
    }

    private SetBidItem convertSetBidsItem(SetBidsItem item) {
        return new SetBidItem()
                .withId(item.getId())
                .withAdGroupId(item.getAdGroupId())
                .withCampaignId(item.getCampaignId())
                .withPriceSearch(convertToDbPrice(ifNotNull(item.getAverageCpc(), JAXBElement::getValue)))
                .withPriceContext(convertToDbPrice(ifNotNull(item.getAverageCpa(), JAXBElement::getValue)))
                .withAutobudgetPriority(convertStrategyPriority(item.getStrategyPriority()));
    }

}
