package ru.yandex.direct.api.v5.entity.smartadtargets.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.smartadtargets.SmartAdTargetUpdateItem;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.api.v5.common.ConverterUtils;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.api.v5.common.ConverterUtils.convertToDbPrice;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.FUNNEL_BY_AUDIENCE;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.RequestConverter.getPerformanceFilterConditionsForUpdate;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.RequestConverter.getTab;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
@Service
public class UpdateRequestConverterService {

    private final FeedService feedService;
    private final PerformanceFilterStorage performanceFilterStorage;

    @Autowired
    public UpdateRequestConverterService(FeedService feedService, PerformanceFilterStorage performanceFilterStorage) {
        this.feedService = feedService;
        this.performanceFilterStorage = performanceFilterStorage;
    }

    private static BigDecimal getPrice(@Nullable JAXBElement<Long> val) {
        return Optional.ofNullable(val)
                // Если в запросе в поле с ценой записан null, то значит надо убрать цену с фильра (приравнять к нулю),
                // чтобы цена бралась из стратегии.
                .map(e -> e.isNil() ? CommonConverters.PRICE_OFF : e.getValue())
                .map(price -> convertToDbPrice(price))
                .orElse(null);
    }

    public List<PerformanceFilter> convertItemsToModels(ClientId clientId,
                                                        List<SmartAdTargetUpdateItem> smartAdTargets) {
        List<Long> filterId = mapList(smartAdTargets, SmartAdTargetUpdateItem::getId);
        Map<Long, Feed> feedByAdGroupId = feedService.getFeedByPerformanceFilterId(clientId, filterId);
        return mapList(smartAdTargets, item -> convertItemToModel(feedByAdGroupId, item));
    }

    private PerformanceFilter convertItemToModel(Map<Long, Feed> feedByAdGroupId, SmartAdTargetUpdateItem item) {
        List<PerformanceFilterCondition> conditions = convertToConditions(feedByAdGroupId, item);
        PerformanceFilterTab tab = getTab(conditions);

        TargetFunnel targetFunnel = ifNotNull(item.getAudience(), FUNNEL_BY_AUDIENCE::get);
        Integer autoBudgetPriority = ifNotNull(item.getStrategyPriority(), ConverterUtils::convertStrategyPriority);
        BigDecimal priceCpc = getPrice(item.getAverageCpc());
        BigDecimal priceCpa = getPrice(item.getAverageCpa());

        return new PerformanceFilter()
                .withPerfFilterId(item.getId())
                .withName(item.getName())
                .withTargetFunnel(targetFunnel)
                .withPriceCpc(priceCpc)
                .withPriceCpa(priceCpa)
                .withAutobudgetPriority(autoBudgetPriority)
                .withConditions(conditions)
                .withTab(tab);
    }

    @Nullable
    @SuppressWarnings("rawtypes")
    private List<PerformanceFilterCondition> convertToConditions(Map<Long, Feed> feedByFilterId,
                                                                 SmartAdTargetUpdateItem item) {
        if (item.getConditions() == null) {
            return null;
        }
        Feed feed = feedByFilterId.get(item.getId());
        // Чтобы для кейса, когда фильтр уже удалён, не пятисотить, а возвращать ошибку валидации.
        if (feed == null) {
            return null;
        }
        BusinessType businessType = feed.getBusinessType();
        FeedType feedType = feed.getFeedType();
        Source source = feed.getSource();
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType, source);
        return getPerformanceFilterConditionsForUpdate(filterSchema, item.getAvailableItemsOnly(),
                item.getConditions());
    }

}
