package ru.yandex.direct.api.v5.entity.smartadtargets.validation;

import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.yandex.direct.api.v5.smartadtargets.SetBidsItem;
import com.yandex.direct.api.v5.smartadtargets.SetBidsRequest;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.maxAdGroupsBidsPerRequest;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.maxCampBidsPerRequest;
import static ru.yandex.direct.api.v5.entity.bids.validation.BidsDefectTypes.maxKeywordBidsPerRequest;
import static ru.yandex.direct.api.v5.validation.DefectTypes.mixedTypes;
import static ru.yandex.direct.api.v5.validation.SetBidsConstraints.bidsListSizeMaxLimit;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Component
@ParametersAreNonnullByDefault
public class SetBidsValidationService {

    private static final String BIDS_FIELD = SetBidsRequest.PropInfo.BIDS.propertyName;

    public static final int MAX_BID_IDS_PER_REQUEST = 10000;
    public static final int MAX_BID_ADGROUPIDS_PER_REQUEST = 1000;
    public static final int MAX_BID_CAMPAIGNIDS_PER_REQUEST = 100;

    private static final List<IdGetter> ID_GETTERS = ImmutableList.of(SetBidsItem::getId,
            SetBidsItem::getAdGroupId, SetBidsItem::getCampaignId);

    public ValidationResult<SetBidsRequest, DefectType> validate(SetBidsRequest setBidsRequest) {
        ItemValidationBuilder<SetBidsRequest, DefectType> vb = ItemValidationBuilder.of(setBidsRequest);
        vb.list(setBidsRequest.getBids(), BIDS_FIELD)
                .check(eachNotNull())
                .check(bidsListSizeMaxLimit(SetBidsItem::getId, MAX_BID_IDS_PER_REQUEST,
                        t -> maxKeywordBidsPerRequest(),
                        SetBidsItem::getAdGroupId, MAX_BID_ADGROUPIDS_PER_REQUEST,
                        t -> maxAdGroupsBidsPerRequest(),
                        SetBidsItem::getCampaignId, MAX_BID_CAMPAIGNIDS_PER_REQUEST,
                        t -> maxCampBidsPerRequest()), When.isValid())
                .check(fromPredicate(this::everyItemHasSingleIdOfTheSameType, mixedTypes()), When.isValid());
        return vb.getResult();
    }

    private boolean everyItemHasSingleIdOfTheSameType(List<SetBidsItem> items) {
        long itemsCount = items.size();
        long totalIdCount = 0L;
        for (IdGetter idGetter : ID_GETTERS) {
            long idsCount = getIdsCount(items, idGetter);
            if (idsCount != 0L && idsCount != itemsCount) {
                return false;
            }
            totalIdCount += idsCount;
        }
        return totalIdCount == itemsCount;
    }

    private static long getIdsCount(List<SetBidsItem> items, IdGetter idGetter) {
        return items.stream()
                .map(idGetter::apply)
                .filter(Objects::nonNull)
                .count();
    }

    @FunctionalInterface
    public interface IdGetter {
        Long apply(SetBidsItem item);
    }

}
