package ru.yandex.direct.api.v5.entity.turbopages.delegate;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.turbopages.GetRequest;
import com.yandex.direct.api.v5.turbopages.GetResponse;
import com.yandex.direct.api.v5.turbopages.TurboPageFieldEnum;
import com.yandex.direct.api.v5.turbopages.TurboPageGetItem;
import com.yandex.direct.api.v5.turbopages.TurboPagesSelectionCriteria;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.turbopages.TurboLandingContainer;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.lang.Integer.max;
import static java.lang.Integer.min;
import static java.util.Collections.emptyList;
import static java.util.Objects.isNull;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.entity.turbopages.TurboPagesEndpoint.PATH_CONVERTER;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxIdsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/turbopages/get-docpage/
 */
@Component
@ParametersAreNonnullByDefault
public class GetTurboPagesDelegate extends
        GetApiServiceDelegate<GetRequest, GetResponse, TurboPageFieldEnum, TurboPagesSelectionCriteria, TurboLandingContainer>
{
    private static final int MAX_BOUND_HREFS = 50;
    private final TurboLandingService turboLandingService;
    private final EnumPropertyFilter<TurboPageFieldEnum> propertyFilter;
    private final ResultConverter resultConverter;

    @Autowired
    public GetTurboPagesDelegate(ApiAuthenticationSource auth,
            TurboLandingService turboLandingService,
            PropertyFilter propertyFilter,
            ResultConverter resultConverter)
    {
        super(PATH_CONVERTER, auth);

        this.turboLandingService = turboLandingService;
        this.resultConverter = resultConverter;
        this.propertyFilter = EnumPropertyFilter.from(TurboPageFieldEnum.class, propertyFilter);
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest externalRequest) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(externalRequest);

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes);

        vb.item(externalRequest.getSelectionCriteria(), "SelectionCriteria")
                .checkBy(this::validateSelectionCriteria, When.notNull());

        return vb.getResult();
    }

    private ValidationResult<TurboPagesSelectionCriteria, DefectType> validateSelectionCriteria(
            TurboPagesSelectionCriteria selectonCriteria)
    {
        ItemValidationBuilder<TurboPagesSelectionCriteria, DefectType> vb = ItemValidationBuilder.of(selectonCriteria);

        vb.item(selectonCriteria.getIds(), "Ids")
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT), maxIdsInSelection(), When.notNull());
        vb.item(selectonCriteria.getBoundWithHrefs(), "Hrefs")
                .check(maxListSize(MAX_BOUND_HREFS), maxElementsInSelection(MAX_BOUND_HREFS));

        return vb.getResult();
    }

    @Override
    public TurboPagesSelectionCriteria extractSelectionCriteria(GetRequest externalRequest) {
        return externalRequest.getSelectionCriteria();
    }

    @Override
    public Set<TurboPageFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public List<TurboLandingContainer> get(
            GenericGetRequest<TurboPageFieldEnum, TurboPagesSelectionCriteria> getRequest)
    {
        ClientId clientId = auth.getChiefSubclient().getClientId();
        var selectionCriteria = getRequest.getSelectionCriteria();
        LimitOffset limitOffset = getRequest.getLimitOffset();

        if (isNull(selectionCriteria)) {
            return getById(clientId, emptyList(), limitOffset);
        }

        List<Long> ids = selectionCriteria.getIds();
        List<String> boundHrefs = selectionCriteria.getBoundWithHrefs();

        if (!boundHrefs.isEmpty()) {
            return getByBoundHrefs(clientId, ids, boundHrefs, limitOffset);
        } else {
            return getById(clientId, ids, limitOffset);
        }
    }

    @Override
    public GetResponse convertGetResponse(List<TurboLandingContainer> result, Set<TurboPageFieldEnum> requestedFields,
            @Nullable Long limitedBy)
    {
        List<TurboPageGetItem> getItems = mapList(result, this::convertTurbolandings);
        propertyFilter.filterProperties(getItems, requestedFields);

        return new GetResponse()
                .withTurboPages(getItems)
                .withLimitedBy(limitedBy);
    }

    private List<TurboLandingContainer> getById(ClientId clientId, List<Long> ids, LimitOffset limitOffset) {
        return StreamEx.of(turboLandingService.getClientTurboLandingsById(clientId, ids,
                limitOffset))
                .map(TurboLandingContainer::new)
                .toList();
    }

    private List<TurboLandingContainer> getByBoundHrefs(ClientId clientId, List<Long> ids, List<String> boundHrefs,
            LimitOffset limitOffset)
    {
        Long clientIdLong = clientId.asLong();
        Map<String, TurboLanding> turboLandingsByHref = turboLandingService.findTurboLandingsByUrl(boundHrefs);
        List<TurboLandingContainer> result = EntryStream.of(turboLandingsByHref)
                .filterValues(t -> clientIdLong.equals(t.getClientId()))
                .filterValues(requestedIdsOnly(ids))
                .map(TurboLandingContainer::new)
                .sortedBy(TurboLanding::getId)
                .toList();

        return cutList(result, limitOffset.offset(), limitOffset.limit());
    }

    private static <T> List<T> cutList(List<T> list, int offset, int limit) {
        if (list.isEmpty() || limit < 1 || offset >= list.size()) {
            return emptyList();
        }
        offset = max(0, offset);
        int rightIdx = min(list.size(), limit + offset);

        return list.subList(offset, rightIdx);
    }

    private Predicate<TurboLanding> requestedIdsOnly(@Nullable List<Long> ids) {
        return t -> ids.isEmpty() || ids.contains(t.getId());
    }

    private TurboPageGetItem convertTurbolandings(TurboLandingContainer turbo) {
        return new TurboPageGetItem()
                .withId(turbo.getId())
                .withName(turbo.getName())
                .withHref(turbo.getUrl())
                .withPreviewHref(turbo.getPreviewHref())
                .withTurboSiteHref(turbo.getTurboSiteHref())
                .withBoundWithHref(turbo.getBoundHref());
    }
}
