package ru.yandex.direct.api.v5.entity.vcards.delegate;

import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.yandex.direct.api.v5.vcards.AddRequest;
import com.yandex.direct.api.v5.vcards.AddResponse;
import com.yandex.direct.api.v5.vcards.InstantMessenger;
import com.yandex.direct.api.v5.vcards.MapPoint;
import com.yandex.direct.api.v5.vcards.Phone;
import com.yandex.direct.api.v5.vcards.VCardAddItem;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.BaseApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.vcards.VCardsEndpoint;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.api.v5.entity.vcards.Constants.MAX_IDS_COUNT_PER_ADD_REQUEST;
import static ru.yandex.direct.api.v5.entity.vcards.VCardsEndpoint.DEFECT_PRESENTATIONS;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsPerRequest;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.common.util.TextUtils.smartStrip;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Добавить визитные карточки
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/vcards/add-docpage/
 */
@Component
@ParametersAreNonnullByDefault
public class AddVCardsDelegate
        extends BaseApiServiceDelegate<AddRequest, AddResponse, List<VCardAddItem>, ApiResult<Long>> {
    private static final Splitter WORKTIME_SPLITTER = Splitter.on(';');

    private final VcardService vcardService;
    private final ResultConverter resultConverter;

    @Autowired
    public AddVCardsDelegate(
            ApiAuthenticationSource auth,
            VcardService vcardService,
            ResultConverter resultConverter) {
        super(VCardsEndpoint.PATH_CONVERTER, auth);
        this.vcardService = vcardService;
        this.resultConverter = resultConverter;
    }

    /**
     * Преобразуем режим работы организации в формат ожидаемый совместимый с perl-ом
     *
     * @param worktime день_с;день_по;час_с;минуты_с;час_до;мин_до(;день_с;день_по;час_с;минуты_с;час_до;мин_до)*
     * @return Строку в формате день_с#день_по#час_с#минуты_с#час_до#мин_до(;день_с#день_по#час_с#минуты_с#час_до#мин_до)*
     */
    @Nullable
    static String toVcardWorktime(@Nullable String worktime) {
        if (StringUtils.isEmpty(worktime)) {
            return worktime;
        }

        if (worktime.charAt(worktime.length() - 1) == ';') {
            worktime = worktime.substring(0, worktime.length() - 1);
        }

        return StreamEx.of(
                Iterables.partition(WORKTIME_SPLITTER.split(worktime), 6)
                        .spliterator())
                .map(p -> String.join("#", p))
                .collect(joining(";"));
    }

    @Nullable
    static ru.yandex.direct.core.entity.vcard.model.Phone toVcardPhone(@Nullable Phone phone) {
        return Optional.ofNullable(phone)
                .map(p -> new ru.yandex.direct.core.entity.vcard.model.Phone()
                        .withCountryCode(smartStrip(p.getCountryCode()))
                        .withCityCode(smartStrip(p.getCityCode()))
                        .withPhoneNumber(smartStrip(p.getPhoneNumber()))
                        .withExtension(smartStrip(p.getExtension())))
                .orElse(null);
    }

    @Nullable
    static ru.yandex.direct.core.entity.vcard.model.InstantMessenger toVcardInstantMessenger(
            @Nullable InstantMessenger messenger) {
        return Optional.ofNullable(messenger)
                .map(im -> new ru.yandex.direct.core.entity.vcard.model.InstantMessenger()
                        .withType(smartStrip(im.getMessengerClient()))
                        .withLogin(smartStrip(im.getMessengerLogin())))
                .orElse(null);
    }

    @Nullable
    static PointOnMap toVcardPointOnMap(@Nullable MapPoint pointOnMap) {
        return Optional.ofNullable(pointOnMap)
                .map(p -> new PointOnMap().withX(p.getX())
                        .withY(p.getY())
                        .withX1(p.getX1())
                        .withY1(p.getY1())
                        .withX2(p.getX2())
                        .withY2(p.getY2()))
                .orElse(null);
    }

    static Vcard toVcard(VCardAddItem item) {
        return new Vcard()
                .withCampaignId(item.getCampaignId())
                .withCountry(smartStrip(item.getCountry()))
                .withCity(smartStrip(item.getCity()))
                .withCompanyName(smartStrip(item.getCompanyName()))
                .withWorkTime(toVcardWorktime(smartStrip(item.getWorkTime())))
                .withPhone(toVcardPhone(item.getPhone()))
                .withStreet(smartStrip(item.getStreet()))
                .withHouse(smartStrip(item.getHouse()))
                .withBuild(smartStrip(item.getBuilding()))
                .withApart(smartStrip(item.getApartment()))
                .withInstantMessenger(toVcardInstantMessenger(item.getInstantMessenger()))
                .withExtraMessage(smartStrip(item.getExtraMessage()))
                .withEmail(smartStrip(item.getContactEmail()))
                .withOgrn(smartStrip(item.getOgrn()))
                .withMetroId(item.getMetroStationId())
                .withManualPoint(toVcardPointOnMap(item.getPointOnMap()))
                .withContactPerson(smartStrip(item.getContactPerson()));
    }

    ValidationResult<List<VCardAddItem>, DefectType> validate(List<VCardAddItem> vcards) {
        return ListValidationBuilder.<VCardAddItem, DefectType>of(vcards)
                .check(notNull())
                .check(maxListSize(MAX_IDS_COUNT_PER_ADD_REQUEST),
                        maxElementsPerRequest(MAX_IDS_COUNT_PER_ADD_REQUEST),
                        When.isValid())
                .check(eachNotNull(), When.isValid())
                .getResult();
    }

    @Override
    public List<VCardAddItem> convertRequest(AddRequest externalRequest) {
        return externalRequest.getVCards();
    }

    @Override
    public ApiResult<List<ApiResult<Long>>> processRequest(List<VCardAddItem> vcards) {
        ValidationResult<List<VCardAddItem>, DefectType> validationResult = validate(vcards);
        if (validationResult.hasAnyErrors()) {
            return ApiResult.broken(
                    validationResult.flattenErrors(),
                    validationResult.flattenWarnings());
        }

        MassResult<Long> result = vcardService.addVcardsPartial(
                mapList(vcards, AddVCardsDelegate::toVcard),
                auth.getOperator().getUid(),
                auth.getChiefSubclient().getClientId()
        );

        return resultConverter.toApiMassResult(result, DEFECT_PRESENTATIONS);
    }

    @Override
    public AddResponse convertResponse(ApiResult<List<ApiResult<Long>>> result) {
        return new AddResponse()
                .withAddResults(resultConverter.toActionResults(result, apiPathConverter));
    }
}
