package ru.yandex.direct.api.v5.entity.vcards.delegate;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.direct.api.v5.general.IdsCriteria;
import com.yandex.direct.api.v5.vcards.GetRequest;
import com.yandex.direct.api.v5.vcards.GetResponse;
import com.yandex.direct.api.v5.vcards.VCardFieldEnum;
import com.yandex.direct.api.v5.vcards.VCardGetItem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.common.EnumPropertyFilter;
import ru.yandex.direct.api.v5.common.validation.GetRequestGeneralValidator;
import ru.yandex.direct.api.v5.converter.ResultConverter;
import ru.yandex.direct.api.v5.entity.GenericGetRequest;
import ru.yandex.direct.api.v5.entity.GetApiServiceDelegate;
import ru.yandex.direct.api.v5.entity.vcards.VCardsEndpoint;
import ru.yandex.direct.api.v5.entity.vcards.converter.VCardConverter;
import ru.yandex.direct.api.v5.result.ApiResult;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.common.util.PropertyFilter;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.VcardService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MAX_IDS_COUNT;
import static ru.yandex.direct.api.v5.common.constants.GetRequestCommonConstants.DEFAULT_MIN_IDS_COUNT;
import static ru.yandex.direct.api.v5.validation.DefectTypes.maxElementsInSelection;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.eachNotNull;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.maxListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.minListSize;
import static ru.yandex.direct.api.v5.validation.constraints.Constraints.notNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Получить визитные карточки
 * <p>
 * https://tech.yandex.ru/direct/doc/ref-v5/vcards/get-docpage/
 */
@Component
@ParametersAreNonnullByDefault
public class GetVCardsDelegate extends GetApiServiceDelegate<GetRequest, GetResponse, VCardFieldEnum, List<Long>, Vcard> {
    private static final Logger logger = LoggerFactory.getLogger(GetVCardsDelegate.class);

    private final VcardService vcardService;
    private final VCardConverter vcardConverter;
    private final EnumPropertyFilter<VCardFieldEnum> propertyFilter;
    private final ResultConverter resultConverter;

    @Autowired
    public GetVCardsDelegate(ApiAuthenticationSource auth,
                             VcardService vcardService, PropertyFilter propertyFilter,
                             ResultConverter resultConverter) {
        super(VCardsEndpoint.PATH_CONVERTER, auth);

        this.vcardService = vcardService;
        this.vcardConverter = new VCardConverter();
        this.propertyFilter = EnumPropertyFilter.from(VCardFieldEnum.class, propertyFilter);
        this.resultConverter = resultConverter;
    }

    @Override
    public ValidationResult<GetRequest, DefectType> validateRequest(GetRequest request) {
        ItemValidationBuilder<GetRequest, DefectType> vb = ItemValidationBuilder.of(request);

        vb.checkBy(GetRequestGeneralValidator::validateRequestWithDefectTypes);

        vb.item(request.getSelectionCriteria(), "SelectionCriteria")
                .checkBy(this::validateSelectionCriteria, When.notNull());

        return vb.getResult();
    }

    private ValidationResult<IdsCriteria, DefectType> validateSelectionCriteria(IdsCriteria selectionCriteria) {
        ItemValidationBuilder<IdsCriteria, DefectType> vb = ItemValidationBuilder.of(selectionCriteria);

        vb.item(selectionCriteria.getIds(), "Ids")
                .check(notNull())
                .check(minListSize(DEFAULT_MIN_IDS_COUNT))
                .check(maxListSize(DEFAULT_MAX_IDS_COUNT),
                        maxElementsInSelection(DEFAULT_MAX_IDS_COUNT)) // // отдельная ошибка создана из-за текста детализации ошибки, ожидаемой в автотестах, вместо maxIdsInSelection())
                .check(eachNotNull());

        return vb.getResult();
    }

    @Override
    public List<Long> extractSelectionCriteria(GetRequest externalRequest) {
        logger.debug("extract ids");
        IdsCriteria selectionCriteria = externalRequest.getSelectionCriteria();
        return selectionCriteria != null && !selectionCriteria.getIds().isEmpty() ? selectionCriteria.getIds() : null;
    }

    @Override
    public Set<VCardFieldEnum> extractFieldNames(GetRequest externalRequest) {
        return new HashSet<>(externalRequest.getFieldNames());
    }

    @Override
    public List<Vcard> get(GenericGetRequest<VCardFieldEnum, List<Long>> getRequest) {
        logger.debug("get vcards");
        long operatorUid = auth.getOperator().getUid();
        ClientId clientId = auth.getChiefSubclient().getClientId();
        List<Long> ids = getRequest.getSelectionCriteria();
        return vcardService.getVcards(operatorUid, clientId, ids, getRequest.getLimitOffset());
    }

    @Override
    public List<Long> returnedCampaignIds(ApiResult<List<Vcard>> result) {
        return mapList(result.getResult(), Vcard::getCampaignId);
    }

    @Override
    public GetResponse convertGetResponse(List<Vcard> result, Set<VCardFieldEnum> requestedFields,
                                          @Nullable Long limitedBy) {
        logger.debug("convert response");

        List<VCardGetItem> getItems = mapList(result, vcardConverter::convert);
        propertyFilter.filterProperties(getItems, requestedFields);

        return new GetResponse()
                .withLimitedBy(limitedBy)
                .withVCards(getItems);
    }
}
