package ru.yandex.direct.api.v5.logging;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Joiner;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.SystemUtils;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Объект, в котором собирается разнородная информация о текущем запросе для записи в логи запросов апи
 * Доступен отовсюду через ApiContextHolder.get().getApiLogRecord()
 * <p>
 * Именования json-аттрибутов совместимые с перловым Директом
 */
@JsonAutoDetect(getterVisibility = JsonAutoDetect.Visibility.NONE)
public class ApiLogRecord {
    private static final DateTimeFormatter LOG_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private final long startNanotime;
    private final int version = 5;
    private Protocol protocol = null;
    private String method = null;
    private String applicationId = "";
    private long requestId;
    private int objectsWithWarningsCount = 0;
    private int objectsWithErrorsCount = 0;
    private LocalDateTime logTime;

    private String operatorLogin = null;
    private long operatorUid = 0;
    private long clientChiefUid = 0;
    private long unitsSpendingUserClientID = 0;

    private int errorStatus = 0;
    private String errorDetails = null;

    private List<Integer> unitsStats = null;
    private Long units = null;

    //JsonProperty здесь потому что для сериализации нужно чтобы поле называлось cid, несмотря на то что содержит список
    @JsonProperty("cid")
    private List<Long> campaignIds = null;
    private List<Long> responseIds = null;
    private String params = null;
    private String response = null;
    private String remoteIp = null;

    public ApiLogRecord() {
        this.startNanotime = System.nanoTime();
        this.requestId = Trace.current().getSpanId();
    }

    @JsonGetter("cmd")
    public String getMethod() {
        return method;
    }

    public ApiLogRecord withMethod(String method) {
        this.method = method;
        return this;
    }

    @JsonGetter("reqid")
    public long getRequestId() {
        return requestId;
    }

    public ApiLogRecord withRequestId(long requestId) {
        this.requestId = requestId;
        return this;
    }

    @JsonIgnore
    public String getOperatorLogin() {
        return operatorLogin;
    }

    public ApiLogRecord withOperatorLogin(String operatorLogin) {
        this.operatorLogin = operatorLogin;
        return this;
    }

    @JsonGetter("uid")
    public long getOperatorUid() {
        return operatorUid;
    }

    public ApiLogRecord withOperatorUid(long operatorUid) {
        this.operatorUid = operatorUid;
        return this;
    }

    @JsonGetter("cluid")
    public long getClientChiefUid() {
        return clientChiefUid;
    }

    public ApiLogRecord withClientChiefUid(Long clientChiefUid) {
        this.clientChiefUid = nvl(clientChiefUid, 0L);
        return this;
    }

    @JsonGetter("api_version")
    public int getVersion() {
        return version;
    }

    public Protocol getProtocol() {
        return protocol;
    }

    @JsonGetter("interface")
    public String getProtocolString() {
        return protocol == null ? null : protocol.name().toLowerCase();
    }

    public ApiLogRecord withProtocol(Protocol protocol) {
        this.protocol = protocol;
        return this;
    }

    @JsonGetter("application_id")
    public String getApplicationId() {
        return applicationId;
    }

    public ApiLogRecord withApplicationId(String applicationId) {
        this.applicationId = applicationId;
        return this;
    }

    @JsonGetter("http_status")
    public long getErrorStatus() {
        return errorStatus;
    }

    public ApiLogRecord withErrorStatus(int errorStatus) {
        this.errorStatus = errorStatus;
        return this;
    }

    @JsonGetter("error_detail")
    public String getErrorDetails() {
        return errorDetails;
    }

    public ApiLogRecord withErrorDetails(String errorDetails) {
        this.errorDetails = errorDetails;
        return this;
    }

    @JsonGetter("units_spending_user_client_id")
    public long getUnitsSpendingUserClientId() {
        return unitsSpendingUserClientID;
    }

    public ApiLogRecord withUnitsSpendingUserClientId(long unitsSpendingUserClientID) {
        this.unitsSpendingUserClientID = unitsSpendingUserClientID;
        return this;
    }

    @JsonGetter("units_stats")
    public String getUnitsStatsString() {
        return unitsStats == null
                ? null
                : "[" + Joiner.on(',').join(unitsStats) + "]";
    }

    public List<Integer> getUnitsStats() {
        return unitsStats;
    }

    public ApiLogRecord withUnitsStats(List<Integer> unitsStats) {
        this.unitsStats = unitsStats;
        return this;
    }

    @JsonGetter("units")
    public Long getUnits() {
        return units;
    }

    public ApiLogRecord withUnits(Long units) {
        this.units = units;
        return this;
    }

    public List<Long> getCampaignIds() {
        return campaignIds;
    }

    public ApiLogRecord withCampaignIds(List<Long> campaignIds) {
        this.campaignIds = campaignIds;
        return this;
    }

    @JsonGetter("response_ids")
    public List<Long> getResponseIds() {
        return responseIds;
    }

    public ApiLogRecord withResponseIds(List<Long> responseIds) {
        this.responseIds = responseIds;
        return this;
    }

    @JsonGetter("param")
    public String getParams() {
        return params;
    }

    public ApiLogRecord withParams(String params) {
        this.params = params;
        return this;
    }

    @JsonGetter("response")
    public String getResponse() {
        return response;
    }

    public ApiLogRecord withResponse(String response) {
        this.response = response;
        return this;
    }

    @JsonGetter("ip")
    public String getRemoteIp() {
        return remoteIp;
    }

    public ApiLogRecord withRemoteIp(String remoteIp) {
        this.remoteIp = remoteIp;
        return this;
    }

    @JsonGetter("host")
    public String getHost() {
        return SystemUtils.hostname();
    }

    @JsonGetter("proc_id")
    public long getProcId() {
        return 0L;
    }

    @JsonGetter("runtime")
    public double getRunTime() {
        return (double) (System.nanoTime() - startNanotime) / TimeUnit.SECONDS.toNanos(1);
    }

    public LocalDateTime getLogTime() {
        return logTime;
    }

    public void fixLogTime() {
        this.logTime = LocalDateTime.now();
    }

    @JsonGetter("logtime")
    public String getLogTimeString() {
        if (logTime == null) {
            fixLogTime();
        }
        return logTime.format(LOG_TIME_FORMATTER);
    }

    @JsonGetter("log_type")
    public String getLogType() {
        return "api";
    }

    @JsonGetter("sleeptime")
    public long getSleepTime() {
        return 0;
    }

    @JsonGetter("warning_object_count")
    public int getObjectsWithWarningsCount() {
        return objectsWithWarningsCount;
    }

    public ApiLogRecord withObjectsWithWarningsCount(int objectsWithWarningsCount) {
        this.objectsWithWarningsCount = objectsWithWarningsCount;
        return this;
    }

    @JsonGetter("error_object_count")
    public int getObjectsWithErrorsCount() {
        return objectsWithErrorsCount;
    }

    public ApiLogRecord withObjectsWithErrorsCount(int objectsWithErrorsCount) {
        this.objectsWithErrorsCount = objectsWithErrorsCount;
        return this;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("ApiLogRecord{");
        sb.append("startNanotime=").append(startNanotime);
        sb.append(", version=").append(version);
        sb.append(", protocol=").append(protocol);
        sb.append(", method='").append(method).append('\'');
        sb.append(", applicationId='").append(applicationId).append('\'');
        sb.append(", requestId=").append(requestId);
        sb.append(", operatorLogin='").append(operatorLogin).append('\'');
        sb.append(", operatorUid=").append(operatorUid);
        sb.append(", clientChiefUid=").append(clientChiefUid);
        sb.append(", errorStatus=").append(errorStatus);
        sb.append(", errorDetails='").append(errorDetails).append('\'');
        sb.append(", unitsSpendingUserClientID=").append(unitsSpendingUserClientID);
        sb.append(", unitsStats=").append(unitsStats);
        sb.append(", units=").append(units);
        sb.append(", cids=").append(campaignIds);
        sb.append(", responseIds=").append(responseIds);
        sb.append(", params='").append(params).append('\'');
        sb.append(", response='").append(response).append('\'');
        sb.append(", remoteIp='").append(remoteIp).append('\'');
        sb.append(", objectsWithWarningsCount='").append(objectsWithWarningsCount).append('\'');
        sb.append(", objectsWithErrorsCount='").append(objectsWithErrorsCount).append('\'');
        sb.append('}');
        return sb.toString();
    }

    public enum Protocol {
        SOAP,
        JSON
    }
}
