package ru.yandex.direct.api.v5.logging;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.context.ApiContextHolder;

/**
 * Объект ApiLogRecord наполняется из разных мест и должен быть доступен во всех интерсепторах
 * и фильтрах. А сохраняться в логи (в базу и текстовый файл) он должен как можно позже.
 * Поэтому нам недостаточно интерсептора, делаем фильтр.
 */
@Component
public class ApiLoggingFilter implements Filter {
    private static final Logger logger = LoggerFactory.getLogger(ApiLoggingFilter.class);

    private final List<ApiLogger> apiLoggers;
    private final ApiContextHolder apiContextHolder;

    @Autowired
    public ApiLoggingFilter(FileLogger fileLogger, ApiContextHolder apiContextHolder) {
        this.apiLoggers = Arrays.asList(fileLogger);
        this.apiContextHolder = apiContextHolder;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        // no initialization
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        try {
            chain.doFilter(request, response);
        } finally {
            logSafely(apiContextHolder.get().getApiLogRecord());
        }
    }

    @Override
    public void destroy() {
        // no finalization
    }

    private void logSafely(ApiLogRecord rec) {
        for (ApiLogger apiLogger : apiLoggers) {
            try {
                apiLogger.log(rec);
            } catch (Exception e) {
                logger.error("Can't log api request to {}", apiLogger.getClass().getName(), e);
            }
        }
    }

}
