package ru.yandex.direct.api.v5.logging;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.direct.api.v5.general.ActionResult;
import com.yandex.direct.api.v5.general.ApiExceptionMessage;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.EndpointInterceptor;

import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.security.DirectApiPreAuthentication;
import ru.yandex.direct.api.v5.ws.ApiMessage;
import ru.yandex.direct.api.v5.ws.WsUtils;
import ru.yandex.direct.api.v5.ws.annotation.ApiMethod;
import ru.yandex.direct.api.v5.ws.json.JsonMessage;
import ru.yandex.direct.api.v5.ws.soap.SoapMessage;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.units.api.UnitsBalance;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.text.StringModifier;
import ru.yandex.misc.reflection.ReflectionUtils;

import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Логгирование запросов в API - основная часть функциональности
 */
@Component
public class ApiLoggingInterceptor implements EndpointInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(ApiLoggingInterceptor.class);

    // кеш для геттеров поля со списком ActionResult из сгенерированных объектов API
    private final ConcurrentHashMap<Class, Optional<Method>> resultsGettersMap = new ConcurrentHashMap<>();

    // заменяем приватные данные звёздочками
    // наверное это можно делать не регекспом, ны быстро простой надёжный способ не нашёлся
    public static final StringModifier JSON_SENSITIVE_DATA_MODIFIER = new StringModifier.Builder()
            .withRegexpReplaceAllRule(
                    Pattern.compile(
                            "(\"password\"\\s*:\\s*\")([^\"]{4,20})(\")",
                            Pattern.CASE_INSENSITIVE),
                    m -> m.group(1) + StringUtils.repeat("*", m.group(2).length()) + m.group(3)
            )
            .build();

    private final ApiContextHolder apiContextHolder;
    private final ObjectMapper objectMapper;
    private final AffectedCampaignIdsContainer affectedCampaignIdsContainer;

    @Autowired
    public ApiLoggingInterceptor(
            ApiContextHolder apiContextHolder,
            @Qualifier(JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME) ObjectMapper objectMapper,
            AffectedCampaignIdsContainer affectedCampaignIdsContainer) {
        this.apiContextHolder = apiContextHolder;
        this.objectMapper = objectMapper.copy();
        this.affectedCampaignIdsContainer = affectedCampaignIdsContainer;
    }

    @Override
    public boolean handleRequest(MessageContext messageContext, Object endpointObject) {
        ApiLogRecord rec = apiContextHolder.get().getApiLogRecord();

        // получаем ip-адрес
        InetAddress ip = HttpUtil.getRemoteAddress(WsUtils.getHttpServletRequest());
        rec.withRemoteIp(ip.getHostAddress());

        Trace trace = Trace.current();

        ApiMethod methodMeta = WsUtils.getEndpointMethodMeta(endpointObject, ApiMethod.class);
        String method = methodMeta.service() + "." + methodMeta.operation();
        trace.setMethod(method);
        Trace.updateMdc();

        rec.withRequestId(trace.getSpanId())
                .withMethod(method);

        return true;
    }

    @Override
    public boolean handleResponse(MessageContext messageContext, Object endpoint) {
        fillAllData(messageContext, endpoint);
        return true;
    }

    @Override
    public boolean handleFault(MessageContext messageContext, Object endpoint) {
        fillAllData(messageContext, endpoint);
        return true;
    }

    @Override
    public void afterCompletion(MessageContext messageContext, Object endpoint, Exception ex) {
        // do nothing
    }

    private void fillAllData(MessageContext messageContext, Object endpoint) {
        ApiLogRecord rec = apiContextHolder.get().getApiLogRecord();

        fillAuth(rec);
        fillRequestParams(rec, messageContext);
        fillResponse(rec, messageContext, endpoint);
        fillFault(rec, messageContext);
        fillUnits(rec, apiContextHolder.get().getUnitsContext().getUnitsBalance());
        fillCampaignIds(rec);
    }

    /**
     * Получаем CampaignIds из контейнера. Контейнер заполняется в
     * {@link ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint}
     */
    private void fillCampaignIds(ApiLogRecord rec) {
        rec.withCampaignIds(new ArrayList<>(affectedCampaignIdsContainer.getIds()));
    }

    /**
     * Получаем из ответа информацию о случившейся ошибке и пишем в ApiLogRecord
     */
    private void fillFault(ApiLogRecord rec, MessageContext messageContext) {
        ApiMessage msg = (ApiMessage) messageContext.getResponse();
        Object apiFault = msg.getApiFault();
        if (apiFault == null) {
            logger.debug("Fault is null in handleFault");
        } else if (apiFault instanceof ApiExceptionMessage) {
            @SuppressWarnings("unchecked")
            ApiExceptionMessage f = (ApiExceptionMessage) apiFault;
            rec.withErrorStatus(f.getErrorCode())
                    .withErrorDetails(f.getErrorDetail());
        } else {
            logger.error("Unsupported fault: {}", apiFault);
        }
    }

    /**
     * Получаем информацию о запросе и пишем в ApiLogRecord
     */
    private void fillRequestParams(ApiLogRecord rec, MessageContext messageContext) {
        ApiMessage msg = (ApiMessage) messageContext.getRequest();

        if (msg instanceof JsonMessage) {
            rec.withProtocol(ApiLogRecord.Protocol.JSON);
        } else if (msg instanceof SoapMessage) {
            rec.withProtocol(ApiLogRecord.Protocol.SOAP);
        } else {
            logger.warn("Unsupported message format: {}", msg.getClass().getCanonicalName());
        }
    }

    /**
     * Получаем тело ответа и получившиеся идентификаторы, сохраняем в ApiLogRecord
     */
    private void fillResponse(ApiLogRecord rec, MessageContext messageContext, Object endpoint) {
        ApiMessage msg = (ApiMessage) messageContext.getResponse();

        Object resp = msg.getApiResponsePayload();
        if (resp == null) {
            return;
        }
        if (needToLogResponse(endpoint)) {
            try {
                rec.withResponse(serializeResponse(resp));
            } catch (JsonProcessingException e) {
                logger.error("Can't serialize response", e);
            }
        }

        rec.withResponseIds(findResponseIds(resp));
        rec.withObjectsWithWarningsCount(countObjectsWithProp(resp, ActionResult::getWarnings, "warnings"));
        rec.withObjectsWithErrorsCount(countObjectsWithProp(resp, ActionResult::getErrors, "errors"));
    }

    String serializeResponse(Object resp) throws JsonProcessingException {
        return JSON_SENSITIVE_DATA_MODIFIER.makeReplacements(
                objectMapper.writeValueAsString(resp));
    }


    private void fillUnits(ApiLogRecord rec, UnitsBalance unitsBalance) {
        long unitsSpent = 0;
        if (unitsBalance != null) {
            unitsSpent = unitsBalance.spentInCurrentRequest();
        }
        rec.withUnits(unitsSpent);
    }

    @Nullable
    private List<Long> findResponseIds(Object resp) {
        Optional<Method> resultsGetter = resultsGettersMap.computeIfAbsent(resp.getClass(), WsUtils::takeResultsGetter);
        if (!resultsGetter.isPresent()) {
            return null;
        }

        try {
            @SuppressWarnings("unchecked")
            List<ActionResult> list = (List<ActionResult>) resultsGetter.get().invoke(resp);
            return mapList(list, ActionResult::getId);
        } catch (ClassCastException ignored) {
            // В ответе может быть и не ActionResult, и это нормально
        } catch (RuntimeException | IllegalAccessException | InvocationTargetException e) {
            logger.error("Can't get response-ids", e);
        }

        return null;
    }

    <E, P extends Collection<?>> int countObjectsWithProp(
            Object response, Function<? super E, ? extends P> resultElementPropertyGetter, String propName) {
        Optional<Method> resultsGetter =
                resultsGettersMap.computeIfAbsent(response.getClass(), WsUtils::takeResultsGetter);
        if (!resultsGetter.isPresent()) {
            return 0;
        }

        List<E> results = null;
        try {
            //noinspection unchecked
            results = (List<E>) resultsGetter.get().invoke(response);
            return (int) results.stream().map(resultElementPropertyGetter).filter(e -> !e.isEmpty()).count();
        } catch (ClassCastException ignored) {
            // В ответе может быть и не ActionResult, и это нормально.
            if (results != null) {
                // Не получилось по-хорошему – берём рефлексией.
                Optional<Integer> count = tryCountPropsUsingReflection(results, propName);
                if (count.isPresent()) {
                    return count.get();
                }
            }
        } catch (RuntimeException | IllegalAccessException | InvocationTargetException e) {
            logger.error("Can't count objects", e);
        }

        return 0;
    }

    private Optional<Integer> tryCountPropsUsingReflection(List<?> results, String propName) {
        try {
            Function<Object, Collection> extractProp = r ->
                    (Collection) ReflectionUtils.invokeMethod(r, "get" + capitalize(propName));
            int count = (int) results.stream().map(extractProp).filter(e -> !e.isEmpty()).count();
            return Optional.of(count);
        } catch (RuntimeException ignored) {
            return Optional.empty();
        }
    }

    /**
     * Заполняем в ApiLogRecord информацию о авторизации
     */
    private void fillAuth(ApiLogRecord rec) {
        DirectApiPreAuthentication preAuth = apiContextHolder.get().getPreAuthentication();
        if (preAuth != null) {
            rec.withOperatorLogin(preAuth.getOperator().getLogin())
                    .withOperatorUid(preAuth.getOperator().getUid())
                    .withClientChiefUid(preAuth.getChiefSubjectUser().map(User::getUid).orElse(null))
                    .withApplicationId(preAuth.getApplicationId());
        } else {
            logger.info("Can't fill auth, no data");
        }
    }

    /**
     * Нужно ли логгировать ответ. Пока что логика простая - для get ответ не логируем, для остальных методов логируем.
     * Нужно будет сделать отдельную аннотацию, которая позволит переопределять умолчательную стратегию
     */
    private boolean needToLogResponse(Object endpoint) {
        ApiMethod endpointDescription = WsUtils.getEndpointMethodMeta(endpoint, ApiMethod.class);
        return !endpointDescription.operation().equals("get");
    }
}
