package ru.yandex.direct.api.v5.result;

import java.util.Collection;
import java.util.List;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.api.v5.validation.DefectType;
import ru.yandex.direct.core.units.OperationSummary;
import ru.yandex.direct.validation.result.DefectInfo;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;

/**
 * Результат пишущей операции.
 * <p>
 * При успешном выполнении операции результат имеет статус {@link ApiResultState#SUCCESSFUL}
 * и содержит результат (id или более расширенную информацию о выполнении).
 * В случае, если исходные данные не прошли валидацию, результат должен иметь статус
 * {@link ApiResultState#BROKEN}, не содержать значение и должен содержать ошибки валидации
 * в {@link #getErrors()}. В случае, если выполнение операции было отменено результат должен иметь
 * статус {@link ApiResultState#CANCELED} и не содержать ни значений, ни ошибок.
 *
 * @param <R> тип значения в результате
 */
public class ApiResult<R> {
    private final ApiResultState state;
    private final List<DefectInfo<DefectType>> errors;
    private final List<DefectInfo<DefectType>> warnings;
    private R result;

    public ApiResult(R result,
                     List<DefectInfo<DefectType>> errors,
                     List<DefectInfo<DefectType>> warnings,
                     ApiResultState state) {
        this.result = result;
        this.errors = ImmutableList.copyOf(errors != null ? errors : emptyList());
        this.warnings = ImmutableList.copyOf(warnings != null ? warnings : emptyList());
        this.state = state;

        checkConsistency();
    }

    public static <R> ApiResult<R> successful(R result) {
        return successful(result, emptyList());
    }

    public static <R> ApiResult<R> successful(R result, List<DefectInfo<DefectType>> warnings) {
        return new ApiResult<>(result, emptyList(), warnings, ApiResultState.SUCCESSFUL);
    }

    public static <R> ApiResult<R> broken(List<DefectInfo<DefectType>> errors, List<DefectInfo<DefectType>> warnings) {
        return new ApiResult<>(null, errors, warnings, ApiResultState.BROKEN);
    }

    public static <R> ApiResult<R> canceled(List<DefectInfo<DefectType>> warnings) {
        return new ApiResult<>(null, emptyList(), warnings, ApiResultState.CANCELED);
    }

    public int getSuccessfulObjectsCount() {
        if (result instanceof Collection) {
            // result - результат простой операции (как get), где "все элементы успешные"
            return ((Collection) result).size();
        }

        return isSuccessful() ? 1 : 0;
    }

    public int getUnsuccessfulObjectsCount() {
        return 0;
    }

    /**
     * @return {@link OperationSummary} с информацией для снятия баллов за операцию
     */
    public OperationSummary getOperationSummary() {
        if (!isSuccessful()) {
            return OperationSummary.unsuccessful();
        }
        int objectsSuccessCount = getSuccessfulObjectsCount();
        int objectsErrorCount = getUnsuccessfulObjectsCount();
        return OperationSummary.successful(objectsSuccessCount, objectsErrorCount);
    }

    /**
     * @return Fetched object if isSuccessful() == {@code true}, {@code null} otherwise.
     */
    public R getResult() {
        return result;
    }

    /**
     * Возвращает все ошибки
     *
     * @return список ошибок
     */
    public List<DefectInfo<DefectType>> getErrors() {
        return errors;
    }

    /**
     * Возвращает все предупреждения
     *
     * @return список предупреждений
     */
    public List<DefectInfo<DefectType>> getWarnings() {
        return warnings;
    }

    /**
     * Возвращает состояние результата
     */
    public ApiResultState getState() {
        return state;
    }

    /**
     * @return {@code true}, if validation result is positive and fetch is successful
     * and {@code false} if validation result is negative.
     */
    public boolean isSuccessful() {
        return state == ApiResultState.SUCCESSFUL;
    }

    private void checkConsistency() {
        switch (this.state) {
            case BROKEN:
                checkState(!this.errors.isEmpty(), "Broken result must contains errors");
                break;
            case CANCELED:
            case SUCCESSFUL:
                checkState(this.errors.isEmpty(), "Successful or canceled result must not contains errors");
                break;
            default:
                throw new IllegalStateException("Unknown state");
        }
    }
}
