package ru.yandex.direct.api.v5.security;

import java.net.InetAddress;

import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import ru.yandex.direct.api.v5.units.UseOperatorUnitsMode;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthCredentials;

import static ru.yandex.direct.api.v5.units.UseOperatorUnitsMode.AUTO;
import static ru.yandex.direct.api.v5.units.UseOperatorUnitsMode.FALSE;
import static ru.yandex.direct.api.v5.units.UseOperatorUnitsMode.TRUE;

/**
 * Информация о запросе для последующей аутентификации в blackbox и API
 */
public class DirectApiCredentials extends BlackboxOauthCredentials implements Cloneable {

    public static final String CLIENT_LOGIN_HEADER = "Client-Login";
    public static final String FAKE_LOGIN_HEADER = "Fake-Login";
    public static final String USE_OPERATOR_UNITS_HEADER = "Use-Operator-Units";
    public static final String TOKEN_TYPE_HEADER = "Token-Type";
    public static final String TOKEN_TYPE_PERSISTENT_VALUE = "persistent";

    private final String userTicket;
    private final String serviceTicket;
    private final String clientLogin;
    private final String fakeLogin;
    private final UseOperatorUnitsMode useOperatorUnitsMode;
    private final Boolean isTokenPersistent;


    public DirectApiCredentials(HttpServletRequest request,
                                InetAddress userIp) {
        super(request, userIp);
        this.userTicket = HttpUtil.getHeaderValue(request, TvmIntegration.USER_TICKET_HEADER)
                .orElse(null);
        this.serviceTicket = HttpUtil.getHeaderValue(request, TvmIntegration.SERVICE_TICKET_HEADER)
                .orElse(null);
        this.clientLogin = HttpUtil.getHeaderValue(request, CLIENT_LOGIN_HEADER)
                .map(PassportUtils::normalizeLogin)
                .orElse(null);
        this.fakeLogin = HttpUtil.getHeaderValue(request, FAKE_LOGIN_HEADER)
                .map(PassportUtils::normalizeLogin)
                .orElse(null);
        if (HttpUtil.isHeaderEqualsToValue(request, USE_OPERATOR_UNITS_HEADER, TRUE.toString())) {
            this.useOperatorUnitsMode = TRUE;
        } else if (HttpUtil.isHeaderEqualsToValue(request, USE_OPERATOR_UNITS_HEADER, AUTO.toString())) {
            this.useOperatorUnitsMode = AUTO;
        } else {
            this.useOperatorUnitsMode = FALSE;
        }
        this.isTokenPersistent = HttpUtil.isHeaderEqualsToValue(
                request, TOKEN_TYPE_HEADER, TOKEN_TYPE_PERSISTENT_VALUE);
    }

    public DirectApiCredentials(String host, InetAddress userIp, String oauthToken, String clientLogin,
                                String fakeLogin, UseOperatorUnitsMode useOperatorUnitsMode,
                                Boolean isTokenPersistent) {
        super(host, userIp, oauthToken);
        this.userTicket = null;
        this.serviceTicket = null;
        this.clientLogin = clientLogin;
        this.fakeLogin = fakeLogin;
        this.useOperatorUnitsMode = useOperatorUnitsMode;
        this.isTokenPersistent = isTokenPersistent;
    }

    @Nullable
    public String getUserTicket() {
        return userTicket;
    }

    @Nullable
    public String getServiceTicket() {
        return serviceTicket;
    }

    @Nullable
    public String getClientLogin() {
        return clientLogin;
    }

    @Nullable
    public String getFakeLogin() {
        return fakeLogin;
    }

    public UseOperatorUnitsMode getUseOperatorUnitsMode() {
        return useOperatorUnitsMode;
    }

    public Boolean isTokenPersistent() {
        return isTokenPersistent;
    }

    @Override
    public DirectApiCredentials clone() throws CloneNotSupportedException {
        return (DirectApiCredentials) super.clone();
    }
}
