package ru.yandex.direct.api.v5.security.internal;

import java.net.InetAddress;
import java.util.Optional;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.AuthenticationException;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.security.DirectApiCredentials;
import ru.yandex.direct.api.v5.security.DirectApiPreAuthentication;
import ru.yandex.direct.api.v5.security.SecurityErrors;
import ru.yandex.direct.api.v5.security.ServiceAuthInterceptor;
import ru.yandex.direct.api.v5.security.exception.BadCredentialsException;
import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.net.NetRangeUtil;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.service.ApiUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.misc.ip.IpAddress;

import static java.util.Collections.emptyList;

/**
 * Аутентификация по базе Директа. Получение субклиента и главных предствителей
 * <p>
 * Дальнейшая авторизация в API выполняется в {@link ServiceAuthInterceptor}
 */
@Component
public class DirectApiInternalAuthProvider {
    private static final Logger logger = LoggerFactory.getLogger(DirectApiInternalAuthProvider.class);

    private final ApiUserService apiUserService;
    private final UserService userService;
    private final NetAcl netAcl;
    private final ClientService clientService;
    private final EnvironmentType environmentType;
    private final BlackboxClient blackboxClient;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmBlackboxService;

    @Autowired
    public DirectApiInternalAuthProvider(
            ApiUserService apiUserService,
            UserService userService,
            NetAcl netAcl,
            ClientService clientService,
            EnvironmentType environmentType,
            BlackboxClient blackboxClient,
            TvmIntegration tvmIntegration) {
        this.apiUserService = apiUserService;
        this.userService = userService;
        this.netAcl = netAcl;
        this.clientService = clientService;
        this.environmentType = environmentType;
        this.blackboxClient = blackboxClient;
        this.tvmIntegration = tvmIntegration;
        tvmBlackboxService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
    }

    /**
     * Аутентификация оператора
     */
    @Nonnull
    public DirectApiPreAuthentication checkApiAccessAllowed(DirectApiInternalAuthRequest authRequest) {
        DirectApiCredentials credentials = authRequest.getCredentials();

        // Получить оператора, с учётом Fake-Login
        ApiUser operator = getAndCheckOperator(authRequest, credentials);
        ApiUser chiefOperator = apiUserService.getChiefRepFor(operator);

        validateIp(authRequest.getCredentials().getUserIp(), operator);

        logger.debug("Successful auth step1. Operator: {} {}. Chief operator: {} {}. ",
                operator.getLogin(), operator.getRole(),
                chiefOperator.getLogin(), chiefOperator.getRole()
        );

        return new DirectApiPreAuthentication(credentials, authRequest.getApplicationId(), operator, chiefOperator,
                authRequest.getTvmUserTicket(), null, null);
    }

    /**
     * Определение целевого пользователя
     */
    @Nonnull
    public DirectApiPreAuthentication checkClientLoginAccess(DirectApiPreAuthentication authCarrier) {
        ApiUser operator = authCarrier.getOperator();
        ApiUser chiefOperator = authCarrier.getChiefOperator();

        // Определяем клиента
        ApiUser subclient;
        ApiUser chiefSubclient;
        String clientLogin = authCarrier.getDirectApiCredentials().getClientLogin();

        if (clientLogin != null) {
            subclient = getUserByLogin(clientLogin);
            chiefSubclient = apiUserService.getChiefRepFor(subclient);
        } else {
            // возможность аутентификации агенства без Client-Login нужна для агентский сервисов в API
            // дополнительные проверки вынесены в ru.yandex.direct.api.v5.security.ServiceTypeAuthInterceptor
            subclient = operator;
            chiefSubclient = chiefOperator;
        }

        // Пользователь яндекс.агенств не имеет прав на API
        if (clientService.isYaAgencyClient(subclient.getClientId()) && !operator.getRole().isInternal()) {
            throw SecurityErrors.newAccessToApiDeniedForYaAgency();
        }

        logger.debug("Successful login. Operator: {} {}. Chief operator: {} {}. " +
                        "Client: {} {}. Chief client: {} {}.",
                operator.getLogin(), operator.getRole(),
                chiefOperator.getLogin(), chiefOperator.getRole(),
                subclient.getLogin(), subclient.getRole(),
                chiefSubclient.getLogin(), chiefSubclient.getRole());

        return new DirectApiPreAuthentication(authCarrier.getDirectApiCredentials(), authCarrier.getApplicationId(),
                operator, chiefOperator, authCarrier.getTvmUserTicket(), subclient, chiefSubclient);
    }

    private ApiUser getApiUserByUid(Long uid) {
        ApiUser operator = apiUserService.getUser(uid);
        if (operator == null) {
            throw SecurityErrors.newUserUidNotFound(uid);
        }

        // Проверить включенность API
        if (operator.getStatusBlocked()) {
            throw SecurityErrors.newAccessToApiDeniedStatusBlocked();
        }

        return operator;
    }

    // из запроса
    private ApiUser getAndCheckOperator(
            DirectApiInternalAuthRequest authRequest, DirectApiCredentials credentials) {
        ApiUser operator = getApiUserByUid(authRequest.getUid());

        // обработка fake-login
        if (operator.getRole().anyOf(RbacRole.SUPER, RbacRole.SUPERREADER)
                && credentials.getFakeLogin() != null
                && !environmentType.isProductionOrPrestable()) {
            Long fakeUid = userService.getUidByLogin(credentials.getFakeLogin());
            if (fakeUid == null) {
                throw SecurityErrors.newUnknownFakeLogin(credentials.getFakeLogin());
            }

            operator = getApiUserByUid(fakeUid);
        }

        return operator;
    }


    private ApiUser getUser(Long uid) throws AuthenticationException {
        ApiUser user = apiUserService.getUser(uid);
        if (user == null) {
            throw new BadCredentialsException("can not find user with uid " + uid);
        }
        return user;
    }

    private ApiUser getUserByLogin(String login) {
        Long uid = Optional.ofNullable(userService.getUidByLogin(login))
                // В паспорте несколько логинов могут быть привязаны к одному uid'у, в то время как в Директе
                // 1 логин <-> 1 uid. Поэтому делаем фоллбэк на паспорт, чтобы вытащить uid, когда к нему привязаны
                // несколько логинов.
                .orElseGet(() -> getUidByLoginFromBlackbox(login));

        if (uid == null) {
            throw SecurityErrors.newUnknownLoginInClientLogin(login);
        }

        return getUser(uid);
    }

    private Long getUidByLoginFromBlackbox(String passportLogin) {
        String tvmTicket = tvmIntegration.getTicket(tvmBlackboxService);
        IpAddress address = HttpUtil.getRemoteAddressForBlackbox();
        BlackboxCorrectResponse user = blackboxClient.userInfo(address, passportLogin, emptyList(), tvmTicket);
        return user.getUid().map(PassportUid::getUid).getOrNull();
    }

    private void validateIp(InetAddress clientAddress, ApiUser operator) {
        if (operator.getRole().isInternal() && !netAcl.isInternalIp(clientAddress)) {
            logger.warn("internal network violation for {}: {}", operator.getRole(), clientAddress);
            throw SecurityErrors.newAccessToApiDeniedAccessFromExternalIp();
        }

        String allowedIps = operator.getApiAllowedIps();
        if (!(allowedIps == null || allowedIps.isEmpty() || NetRangeUtil.isIpInNetworks(clientAddress, allowedIps))) {
            throw SecurityErrors.newAccessToApiDeniedNotAllowedIp();
        }
    }
}
