package ru.yandex.direct.api.v5.security.ticket;

import java.util.Map;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.security.DirectApiCredentials;
import ru.yandex.direct.api.v5.security.SecurityErrors;
import ru.yandex.direct.api.v5.security.exception.BadCredentialsException;
import ru.yandex.direct.api.v5.security.internal.DirectApiInternalAuthRequest;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.application.service.ApiAppAccessService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static ru.yandex.direct.api.v5.security.SecurityErrors.newUserUidNotFound;

/**
 * Класс для аутентификации по Tvm'ному User-тикету:
 * https://wiki.yandex-team.ru/passport/tvm2/user-ticket/
 */

@Lazy
@Component
public class TvmUserTicketAuthProvider {

    private final ApiAppAccessService apiAppAccessService;
    private final TvmIntegration tvmIntegration;
    private final ShardHelper shardHelper;
    private final Map<TvmService, String> applicationIdByTvmService;

    @Autowired
    public TvmUserTicketAuthProvider(ApiAppAccessService apiAppAccessService,
                                     TvmIntegration tvmIntegration,
                                     ShardHelper shardHelper,
                                     DirectConfig directConfig) {
        this.apiAppAccessService = apiAppAccessService;
        this.tvmIntegration = tvmIntegration;
        this.shardHelper = shardHelper;
        this.applicationIdByTvmService = getApplicationIdByTvmService(directConfig);
    }

    private Map<TvmService, String> getApplicationIdByTvmService(DirectConfig directConfig) {
        DirectConfig branch = directConfig.getBranch("tvm_api_auth");
        Map<String, String> topicForSourcePrefix = branch.asMap();
        return EntryStream.of(topicForSourcePrefix)
                .mapKeys(TvmService::fromStringStrict)
                .toMap();
    }

    public DirectApiInternalAuthRequest authenticate(DirectApiCredentials credentials) {
        String serviceTicket = credentials.getServiceTicket();
        if (StringUtils.isEmpty(serviceTicket)) {
            throw new BadCredentialsException("Сервисный TVM-тикет не задан");
        }
        TvmService tvmService = tvmIntegration.getTvmService(serviceTicket);
        String applicationId = applicationIdByTvmService.get(tvmService);
        if (StringUtils.isEmpty(applicationId)) {
            throw new BadCredentialsException("В конфиг-файле не найден applicationId для указанного TVM-сервиса");
        }
        checkApplicationId(applicationId);
        String userTicket = credentials.getUserTicket();
        if (StringUtils.isEmpty(userTicket)) {
            throw new BadCredentialsException("User-тикет не задан");
        }
        long uid = tvmIntegration.checkUserTicket(userTicket);
        String login = getLogin(uid);
        return new DirectApiInternalAuthRequest(credentials, uid, login, applicationId, userTicket);
    }

    private String getLogin(long uid) {
        try {
            return shardHelper.getLoginByUid(uid);
        } catch (IllegalArgumentException ignore) {
            throw newUserUidNotFound(uid);
        }
    }

    private void checkApplicationId(String applicationId) {
        if (!apiAppAccessService.checkApplicationAccess(applicationId)) {
            throw SecurityErrors.newApplicationIdNotRegistered();
        }
    }

}
