package ru.yandex.direct.api.v5.security.token;

import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.authentication.AccountStatusException;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.security.DirectApiCredentials;
import ru.yandex.direct.api.v5.security.SecurityErrors;
import ru.yandex.direct.api.v5.security.exception.BadCredentialsException;
import ru.yandex.direct.api.v5.security.internal.DirectApiInternalAuthRequest;
import ru.yandex.direct.core.entity.application.service.ApiAppAccessService;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthAuth;
import ru.yandex.direct.web.auth.blackbox.BlackboxOauthAuthProvider;
import ru.yandex.direct.web.auth.blackbox.exception.BadBlackboxCredentialsException;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxOAuthInfo;

@Lazy
@Component
public class DirectApiTokenAuthProvider implements ApiTokenAuthProvider {
    private static final Pattern OAUTH_TOKEN_PATTERN = Pattern.compile("[a-zA-Z\\-\\.\\_\\~\\+\\/0-9]+\\=*");
    static final String DIRECT_API_SCOPE = "direct:api";

    private final BlackboxOauthAuthProvider blackboxOauthAuthProvider;
    private final PersistentTokenAuthProvider persistentTokenAuthProvider;
    private final ApiAppAccessService apiAppAccessService;
    private final ApiContextHolder apiContextHolder;

    @Autowired
    public DirectApiTokenAuthProvider(BlackboxOauthAuthProvider blackboxOauthAuthProvider,
                                      PersistentTokenAuthProvider persistentTokenAuthProvider,
                                      ApiAppAccessService apiAppAccessService,
                                      ApiContextHolder apiContextHolder) {
        this.blackboxOauthAuthProvider = blackboxOauthAuthProvider;
        this.persistentTokenAuthProvider = persistentTokenAuthProvider;
        this.apiAppAccessService = apiAppAccessService;
        this.apiContextHolder = apiContextHolder;
    }

    @Override
    public DirectApiInternalAuthRequest authenticate(DirectApiTokenAuthRequest authRequest) {
        DirectApiCredentials requestInfo = authRequest.getCredentials();
        if (requestInfo.isTokenPersistent()) {
            return authenticateByPersistentToken(authRequest);
        } else {
            return authenticateByOauthToken(authRequest);
        }
    }

    private DirectApiInternalAuthRequest authenticateByOauthToken(DirectApiTokenAuthRequest apiTokenAuthRequest) {
        String token = apiTokenAuthRequest.getCredentials().getOauthToken();

        if (StringUtils.isEmpty(token)) {
            throw SecurityErrors.newAbsentOauthToken();
        }

        if (!OAUTH_TOKEN_PATTERN.matcher(token).matches()) {
            throw SecurityErrors.newInvalidOauthTokenFormat(null);
        }

        BlackboxOauthAuth blackboxOauthAuth = authenticateAtBlackbox(apiTokenAuthRequest);
        String applicationId = blackboxOauthAuth.getOauthResponse().getOAuthInfoOptional()
                .map(BlackboxOAuthInfo::getClientId)
                .orElseThrow(() -> new RuntimeException("no application id in BB response"));
        // DIRECT-80738: Это единственное место, где можно безболезненно достать applicationId даже если мы
        // на ней упали
        apiContextHolder.get().getApiLogRecord().withApplicationId(applicationId);
        List<String> applicationScopes = blackboxOauthAuth.getOauthResponse().getOAuthInfoOptional()
                .map(BlackboxOAuthInfo::getScopes)
                .orElseThrow(() -> new RuntimeException("no application scopes in BB response"));
        checkApplicationScope(applicationScopes);
        checkApplicationId(applicationId);
        return new DirectApiInternalAuthRequest(apiTokenAuthRequest.getCredentials(),
                blackboxOauthAuth.getUid(), blackboxOauthAuth.getPrincipal(), applicationId,
                blackboxOauthAuth.getOauthResponse().getTvmUserTicketOptional().orElse(null));
    }

    private DirectApiInternalAuthRequest authenticateByPersistentToken(DirectApiTokenAuthRequest apiTokenAuthRequest) {
        PersistentTokenAuth persistentTokenAuth = persistentTokenAuthProvider.authenticate(apiTokenAuthRequest);
        return new DirectApiInternalAuthRequest(apiTokenAuthRequest.getCredentials(), persistentTokenAuth.getUid(),
                persistentTokenAuth.getPrincipal(), persistentTokenAuth.getApplicationId(),
                null);
    }

    private BlackboxOauthAuth authenticateAtBlackbox(DirectApiTokenAuthRequest apiTokenAuthRequest) {
        try {
            return blackboxOauthAuthProvider.authenticate(apiTokenAuthRequest);
        } catch (BadCredentialsException | AccountStatusException e) {
            throw SecurityErrors.newExpiredOauthToken(e);
        } catch (BadBlackboxCredentialsException e) {
            throw SecurityErrors.newInvalidOauthTokenFormat(e);
        } catch (RuntimeException e) {
            throw SecurityErrors.newInternalBlackboxError(e);
        }
    }

    private void checkApplicationScope(List<String> scopes) {
        if (!scopes.contains(DIRECT_API_SCOPE)) {
            throw SecurityErrors.newInvalidOauthToken();
        }
    }

    private void checkApplicationId(String applicationId) {
        if (!apiAppAccessService.checkApplicationAccess(applicationId)) {
            throw SecurityErrors.newApplicationIdNotRegistered();
        }
    }
}
