package ru.yandex.direct.api.v5.security.token;

import java.net.InetAddress;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.List;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;
import org.apache.commons.codec.binary.Hex;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.InternalAuthenticationServiceException;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.ApiFaultTranslations;
import ru.yandex.direct.api.v5.security.SecurityErrors;
import ru.yandex.direct.api.v5.security.exception.BadCredentialsException;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.liveresource.LiveResourceFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * <p> Класс для аутентификации по персистентному токену.
 * <p> Персистентные токены - не протухающие токены, нужны для различных внутренних нужд и сервисов
 * <p> Выдергивается токен из пришедшего запроса, шифруется с помощью hmac sha256 и секретного ключа
 * api-persistent-token-key из конфига, и проверяется его наличие в конфиге.
 * <p> Токены в шифрованном виде хранятся в конфиге, задаваемым настройкой persistent_token_conf
 * <p> Так же проверяется ip с которого пришел запрос на соответствие сетям, заданным в конфиге для каждого конкретного
 * персистентного токена
 */

@Component
public class PersistentTokenAuthProvider {

    private static final String API_PERSISTENT_TOKEN_KEY_CONFIG_PATH = "api-persistent-token-key";

    private String apiTokenKey;
    private final ShardHelper shardHelper;
    private NetAcl netAcl;
    private Config persistentTokensConfig;
    private Mac sha256HMAC;

    public PersistentTokenAuthProvider(ShardHelper shardHelper, NetAcl netAcl,
                                       List<String> confUris) {
        this.shardHelper = shardHelper;
        this.netAcl = netAcl;

        List<Config> configs = confUris.stream()
                .map(confUri -> ConfigFactory.parseString(LiveResourceFactory.get(confUri).getContent()))
                .collect(toList());

        List<String> apiTokenKeys = configs.stream()
                .filter(config -> config.hasPath(API_PERSISTENT_TOKEN_KEY_CONFIG_PATH))
                .map(config -> config.getString(API_PERSISTENT_TOKEN_KEY_CONFIG_PATH))
                .collect(toList());

        checkState(apiTokenKeys.size() == 1, "exactly one config has %s", API_PERSISTENT_TOKEN_KEY_CONFIG_PATH);

        this.apiTokenKey = apiTokenKeys.get(0);

        this.persistentTokensConfig = configs.stream()
                .map(config -> config.getConfig("api-persistent-token"))
                .reduce(Config::withFallback)
                .orElseThrow(IllegalStateException::new);

        initSha256HMAC();
    }

    @Autowired
    public PersistentTokenAuthProvider(ShardHelper shardHelper, NetAcl netAcl,
                                       DirectConfig directConfig) {
        this(shardHelper, netAcl, directConfig.getStringList("persistent_token_conf"));
    }

    /**
     * Аутентификация по персистентному токену.
     *
     * @param apiTokenAuthRequest При удачной аутентификации возвращается заполненный объект PersistentTokenAuth
     * @return PersistentTokenAuth
     * @throws BadCredentialsException если
     *                                 <ul>
     *                                 <li>
     *                                 Если для переданного токена нет информации в конфиге
     *                                 </li>
     *                                 <li>
     *                                 Если запрос пришел не из разрешенных сетей
     *                                 </li>
     *                                 </ul>
     */
    public PersistentTokenAuth authenticate(DirectApiTokenAuthRequest apiTokenAuthRequest) {
        String token = apiTokenAuthRequest.getCredentials().getOauthToken();
        if (token == null)
            throw SecurityErrors.newAbsentOauthToken();
        String encodedToken = encode(token);
        if (!persistentTokensConfig.hasPath(encodedToken)) {
            throw SecurityErrors.newPersistentTokenNotFound();
        }
        Config tokenConfig = persistentTokensConfig.getConfig(encodedToken);
        List<String> allowToCollection = mapList(tokenConfig.getList("allow_to").unwrapped(), Object::toString);
        InetAddress userIp = apiTokenAuthRequest.getCredentials().getUserIp();

        if (!netAcl.isIpInNetworks(userIp, allowToCollection)) {
            throw new BadCredentialsException("Доступ возможен только из внутренней сети Яндекса");
        }
        String login = tokenConfig.getString("login");
        String applicationId = tokenConfig.getString("application_id");
        Long uid = shardHelper.getUidByLogin(login);

        return new PersistentTokenAuth(apiTokenAuthRequest.getCredentials(), uid, login, applicationId);
    }

    private void initSha256HMAC() {
        try {
            sha256HMAC = Mac.getInstance("HmacSHA256");
            sha256HMAC.init(new SecretKeySpec(apiTokenKey.getBytes(), "HmacSHA256"));
        } catch (NoSuchAlgorithmException e) {
            throw new InternalAuthenticationServiceException("Algorithm for HmacSHA256 not found", e);
        } catch (InvalidKeyException e) {
            throw new InternalAuthenticationServiceException("Invalid Key for initialization sha256 HMAC algorithm", e);
        }
    }

    private String encode(String data) {
        return Hex.encodeHexString(sha256HMAC.doFinal(data.getBytes()));
    }

    private static ApiFaultTranslations getTranslations() {
        return ApiFaultTranslations.INSTANCE;
    }

}
