package ru.yandex.direct.api.v5.semaphore;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.EndpointInterceptor;

import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.redislock.DistributedLock;
import ru.yandex.direct.redislock.DistributedLockException;
import ru.yandex.direct.redislock.lettuce.LettuceLockBuilder;

@Component
public class ApiLockInterceptor implements EndpointInterceptor {
    public static final String REQUEST_LOCK_ENTRY_NAME =
            ApiLockInterceptor.class.getName() + ".SIMULTANEOUS_CONN_LIMIT_LOCK";

    private static final Logger logger = LoggerFactory.getLogger(ApiLockInterceptor.class);
    private LettuceLockBuilder lockBuilder;

    @Autowired
    DirectConfig config;

    @Autowired
    LettuceConnectionProvider lettuce;

    @Autowired
    ApiAuthenticationSource authSource;

    @PostConstruct
    protected void init() {
        DirectConfig lockConfig = config.getBranch("api-connections-lock");
        int maxLocks = lockConfig.getInt("max-locks");
        long negotiationTimeout = lockConfig.getInt("negotiation-timeout");
        String lockKeyPrefix = lockConfig.getString("lock-key-prefix");
        lockBuilder = LettuceLockBuilder
                .newBuilder(lettuce::getConnection)
                .withMaxLocks(maxLocks)
                .withLockAttemptTimeout(negotiationTimeout)
                .withKeyPrefix(lockKeyPrefix);
    }

    @Override
    public boolean handleRequest(MessageContext messageContext, Object endpoint) throws ApiLockException {
        boolean locked;
        try {
            DistributedLock lock = createLockForClient();
            if (locked = lock.lock()) {
                messageContext.setProperty(REQUEST_LOCK_ENTRY_NAME, lock);
            }
        } catch (Exception ex) {
            logger.error("cant acquire API lock: lock service exception", ex);
            return true;
        }

        if (!locked) {
            throw new ApiLockException();
        }

        return true;
    }

    private DistributedLock createLockForClient() {
        ApiUser apiUser = authSource.getChiefSubclient();
        Long clientId = apiUser.getClientId().asLong();
        DistributedLock lock = apiUser.getConcurrentCalls() == null ?
                lockBuilder.createLock(clientId.toString()) :
                lockBuilder.createLock(clientId.toString(), apiUser.getConcurrentCalls().intValue());
        return lock;
    }

    @Override
    public boolean handleResponse(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public boolean handleFault(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public void afterCompletion(MessageContext messageContext, Object endpoint, Exception ex) throws Exception {
        DistributedLock lock = (DistributedLock) messageContext.getProperty(REQUEST_LOCK_ENTRY_NAME);
        messageContext.removeProperty(REQUEST_LOCK_ENTRY_NAME);
        if (lock != null) {
            try {
                lock.unlock();
            } catch (DistributedLockException lockEx) {
                logger.error("cant release API lock: lock service exception", lockEx);
            }
        }
    }
}
