package ru.yandex.direct.api.v5.semaphore.jvm;

import java.util.concurrent.Semaphore;

import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.EndpointInterceptor;

import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.ws.WsUtils;
import ru.yandex.direct.api.v5.ws.annotation.ApiMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.web.core.semaphore.jvm.JvmSemaphore;
import ru.yandex.direct.web.core.semaphore.jvm.JvmSemaphoreException;

@Component
public class ApiJvmSemaphoreInterceptor implements EndpointInterceptor {
    static final String REQUEST_SEMAPHORE_ENTRY_NAME =
            ApiJvmSemaphoreInterceptor.class.getName() + ".SERVICE_SEMAPHORE";

    private static final Logger logger = LoggerFactory.getLogger(ApiJvmSemaphoreInterceptor.class);

    private final JvmSemaphore semaphore;
    private final ApiContextHolder apiContextHolder;

    public ApiJvmSemaphoreInterceptor(DirectConfig directConfig, ApiContextHolder apiContextHolder) {
        DirectConfig config = directConfig.getBranch("service_semaphore");
        this.semaphore = new JvmSemaphore(config);
        this.apiContextHolder = apiContextHolder;
    }

    @PostConstruct
    protected void init() {
        // pass
    }

    @Override
    public boolean handleRequest(MessageContext messageContext, Object endpoint) {
        ApiMethod methodMeta = WsUtils.getEndpointMethodMeta(endpoint, ApiMethod.class);
        String serviceName = methodMeta.service();

        Semaphore sem;
        try {
            sem = semaphore.tryAcquireAndIncrementStat(serviceName);
        } catch (JvmSemaphoreException e) {
            logger.error("Can't acquire semaphore");
            apiContextHolder.get().setShouldChargeUnitsForRequest(false);
            throw new ApiJvmSemaphoreException();
        }
        if (sem != null) {
            messageContext.setProperty(REQUEST_SEMAPHORE_ENTRY_NAME, sem);
        }

        return true;
    }

    @Override
    public boolean handleResponse(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public boolean handleFault(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public void afterCompletion(MessageContext messageContext, Object endpoint, Exception ex) throws Exception {
        Semaphore sem = (Semaphore) messageContext.getProperty(REQUEST_SEMAPHORE_ENTRY_NAME);
        messageContext.removeProperty(REQUEST_SEMAPHORE_ENTRY_NAME);
        if (sem != null) {
            sem.release();
        }
    }
}
