package ru.yandex.direct.api.v5.service;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.yandex.direct.api.v5.bids.BidActionResult;
import com.yandex.direct.api.v5.general.ActionResultBase;
import com.yandex.direct.api.v5.keywordbids.KeywordBidActionResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.EndpointInterceptor;

import ru.yandex.direct.api.v5.ws.ApiMessage;
import ru.yandex.direct.api.v5.ws.WsUtils;

/**
 * Заменяет пустые списки ошибок и предупреждений в ответе API на null, чтобы клиенты не получали неожиданных пустых
 * списков ошибок (в наших бинах при запросе списка ошбок или предупреждений null автоматически заменяется на пустой
 * список, этот интерсептор это компенсирует)
 */
@Component
public class ApiNoEmptyErrorListsInterceptor implements EndpointInterceptor {
    private final ConcurrentMap<Class, Optional<Method>> resultsGettersMap = new ConcurrentHashMap<>();
    private static final Logger logger = LoggerFactory.getLogger(ApiNoEmptyErrorListsInterceptor.class);

    @Override
    public boolean handleRequest(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public boolean handleResponse(MessageContext messageContext, Object endpoint) throws Exception {
        Object msg = messageContext.getResponse();
        if (!(msg instanceof ApiMessage)) {
            return true;
        }

        Object response = ((ApiMessage) msg).getApiResponsePayload();
        if (response == null) {
            return true;
        }

        Optional<Method> resultsGetter =
                resultsGettersMap.computeIfAbsent(response.getClass(), WsUtils::takeResultsGetter);
        if (!resultsGetter.isPresent()) {
            return true;
        }

        Object results;
        try {
            results = resultsGetter.get().invoke(response);
        } catch (RuntimeException | IllegalAccessException | InvocationTargetException e) {
            logger.error("Can't replace empty lists with nulls", e);
            return true;
        }

        if (List.class.isAssignableFrom(results.getClass())) {
            List<?> resultsList = (List<?>) results;
            if (resultsList.isEmpty() || resultsList.get(0) == null) {
                return true;
            }
            Class<?> cls = resultsList.get(0).getClass();
            if (ActionResultBase.class.isAssignableFrom(cls)) {
                tryCleanActionResult((List<ActionResultBase>) results);
            } else if (KeywordBidActionResult.class.isAssignableFrom(cls)) {
                tryCleanKeywordBidActionResult((List<KeywordBidActionResult>) results);
            } else if (BidActionResult.class.isAssignableFrom(cls)) {
                tryCleanBidActionResult((List<BidActionResult>) results);
            }
        }
        return true;
    }

    private void tryCleanActionResult(List<ActionResultBase> results) {
        for (ActionResultBase result : results) {
            // getErrors/getWarnings всегда возвращают не null, null заменяется на пустой список, меняем обратно,
            // иначе клиентам приходят пустые списки ошибок и предупреждений вмсто их отсутствия
            if (result.getErrors().isEmpty()) {
                result.setErrors(null);
            }
            if (result.getWarnings().isEmpty()) {
                result.setWarnings(null);
            }
        }
    }

    private void tryCleanBidActionResult(List<BidActionResult> results) {
        for (BidActionResult result : results) {
            // getErrors/getWarnings всегда возвращают не null, null заменяется на пустой список, меняем обратно,
            // иначе клиентам приходят пустые списки ошибок и предупреждений вмсто их отсутствия
            if (result.getErrors().isEmpty()) {
                result.setErrors(null);
            }
            if (result.getWarnings().isEmpty()) {
                result.setWarnings(null);
            }
        }
    }

    private void tryCleanKeywordBidActionResult(List<KeywordBidActionResult> results) {
        for (KeywordBidActionResult result : results) {
            // getErrors/getWarnings всегда возвращают не null, null заменяется на пустой список, меняем обратно,
            // иначе клиентам приходят пустые списки ошибок и предупреждений вмсто их отсутствия
            if (result.getErrors().isEmpty()) {
                result.setErrors(null);
            }
            if (result.getWarnings().isEmpty()) {
                result.setWarnings(null);
            }
        }
    }

    @Override
    public boolean handleFault(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public void afterCompletion(MessageContext messageContext, Object endpoint, Exception ex) throws Exception {
    }
}
