package ru.yandex.direct.api.v5.units;

import java.io.IOException;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.context.ApiContext;
import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.context.units.UnitsContext;
import ru.yandex.direct.api.v5.security.DirectApiPreAuthentication;
import ru.yandex.direct.core.units.api.UnitsBalance;

/**
 * Юниты нам бывают нужны до интерсепторов - в момент разбора запроса.
 * <p>
 * Перед обработкой запроса здесь, в фильтре, {@link UnitsContext}, в котором содержится информация
 * о том, с кого будем списывать баллы, укладывается в {@link ApiContext}.
 * <p>
 * Также именно в фильтре происходит запись потраченных баллов в хранилище.
 * <p>
 * Определение цены конкретной операции и простановка заголовков ответа производятся в интерсепторе.
 *
 * @see UnitsContextFactory
 * @see UnitsInterceptor
 */
@Component
@ParametersAreNonnullByDefault
public class UnitsFilter implements Filter {

    private static final Logger logger = LoggerFactory.getLogger(UnitsFilter.class);

    private final ApiUnitsService apiUnitsService;
    private final ApiContextHolder apiContextHolder;
    private final UnitsContextFactory unitsContextFactory;

    @Autowired
    public UnitsFilter(ApiUnitsService apiUnitsService, ApiContextHolder apiContextHolder,
                       UnitsContextFactory unitsContextFactory) {
        this.apiUnitsService = apiUnitsService;
        this.apiContextHolder = apiContextHolder;
        this.unitsContextFactory = unitsContextFactory;
    }

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        // no initialization
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain)
            throws IOException, ServletException {
        ApiContext apiContext = apiContextHolder.get();
        DirectApiPreAuthentication preAuth = apiContext.getPreAuthentication();
        if (preAuth == null) {
            logger.warn("Authentication is null!");
        }

        apiContext.setUnitsContext(unitsContextFactory.createUnitsContext(preAuth));

        try {
            chain.doFilter(request, response);
        } finally {
            updateSpentUnits(apiContext);
        }
    }

    private void updateSpentUnits(ApiContext apiContext) {
        UnitsContext unitsContext = apiContext.getUnitsContext();
        if (unitsContext == null) {
            logger.warn("UnitsContext is null!");
            return;
        }
        UnitsBalance unitsBalance = unitsContext.getUnitsBalance();
        UnitsBalance operatorUnitsBalance = unitsContext.getOperatorUnitsBalance();
        if (unitsBalance != null) {
            apiUnitsService.updateSpent(unitsBalance);
        }
        if (operatorUnitsBalance != null) {
            // если этот баланс принадлежит тому же клиенту, что и unitsBalance, ничего страшного;
            // списание баллов за ошибку произойдёт лишь однажды - здесь.
            apiUnitsService.updateSpent(operatorUnitsBalance);
        }
    }

    @Override
    public void destroy() {
        // no finalization
    }
}
