package ru.yandex.direct.api.v5.units;

import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.EndpointInterceptor;

import ru.yandex.direct.api.v5.context.ApiContext;
import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.context.units.UnitsContext;
import ru.yandex.direct.api.v5.security.DirectApiPreAuthentication;
import ru.yandex.direct.api.v5.units.exception.LackOfUnitsException;
import ru.yandex.direct.api.v5.units.exception.NoPreAuthDataException;
import ru.yandex.direct.api.v5.ws.WsUtils;
import ru.yandex.direct.api.v5.ws.annotation.ApiMethod;
import ru.yandex.direct.core.units.OperationCosts;
import ru.yandex.direct.core.units.api.UnitsBalance;
import ru.yandex.direct.core.units.service.UnitsService;

/**
 * {@link UnitsInterceptor} работает с баллами при обработке пользовательского запроса, осуществляя проверку баланса и
 * списание баллов.
 * <p>
 * {@link UnitsInterceptor} работает в паре с {@link UnitsFilter}, который определяет {@link UnitsContext},
 * хранящий информацию о балансе клиента, и проставляет его в {@link ApiContext}, после чего этот объект
 * становится доступен здесь. {@link UnitsInterceptor} в {@link #handleRequest(MessageContext, Object)} определяет,
 * сколько стоит вызов, и генерирует {@link LackOfUnitsException}, если баллов у текущего unitsHolder'а не хватает.
 * Здесь же в {@link ru.yandex.direct.api.v5.context.ApiContext} проставляются {@link OperationCosts} с описанием
 * стоимостей текущей операции.
 * <p>
 * Ожидается, что при обработке пользовательского запроса {@link OperationCosts} будет использован
 * для вычисления затрат, которые будут списаны с помощью {@link UnitsBalance} из {@link UnitsContext}.
 * <p>
 *
 * @see UnitsService
 */
@Component
public class UnitsInterceptor implements EndpointInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(UnitsInterceptor.class);

    private final ApiUnitsService apiUnitsService;
    private final ApiContextHolder apiContextHolder;
    private final UnitsContextFactory unitsContextFactory;

    @Autowired
    public UnitsInterceptor(ApiUnitsService apiUnitsService, ApiContextHolder apiContextHolder,
                            UnitsContextFactory unitsContextFactory) {
        this.apiUnitsService = apiUnitsService;
        this.apiContextHolder = apiContextHolder;
        this.unitsContextFactory = unitsContextFactory;
    }

    @Override
    public boolean handleRequest(MessageContext messageContext, Object endpoint) throws Exception {
        boolean isUnitsEnough;
        try {
            ApiContext apiContext = apiContextHolder.get();
            DirectApiPreAuthentication preAuth = apiContext.getPreAuthentication();
            OperationCosts costs = getOperationCosts(endpoint);
            apiContext.setOperationCosts(costs);
            int operationCost = costs.getCostForOperation();

            // В режиме "Use-Operator-Units: auto" клиенту может не хватать баллов на операцию. В этом случае
            // необходимо списать баллы с агенства. Для этого создадим UnitsContext заново с учетом стоимости операции.
            UnitsContext unitsContext = unitsContextFactory.createUnitsContext(preAuth, operationCost);
            apiContext.setUnitsContext(unitsContext);

            UnitsBalance unitsBalance = unitsContext.getUnitsBalanceOrFail();

            if (unitsBalance.getLimit() < costs.getMinDailyLimit()) {
                // TODO: другое исключение
                throw new LackOfUnitsException();
            }

            isUnitsEnough = unitsBalance.isAvailable(operationCost);
        } catch (NoPreAuthDataException e) {
            logger.info("Authentication data does not present so unit context is absent too", e);
            // контекста нет, баллы списать не можем, но запрос нужно пропустить дальше,
            // для того, чтобы в интерсепторе авторизации разобраться почему нет данных о пользователе и
            // выкинуть правильную ошибку
            isUnitsEnough = true;
        } catch (RuntimeException e) {
            logger.warn("Error in initializing units for current request", e);
            throw e;
        }
        if (!isUnitsEnough) {
            throw new LackOfUnitsException();
        }
        return true;
    }

    private OperationCosts getOperationCosts(Object endpoint) {
        ApiMethod apiMethod = getEndpointMethodMeta(endpoint);
        String serviceName = apiMethod.service();
        String methodName = apiMethod.operation();
        return apiUnitsService.getCostsForOperation(serviceName, methodName);
    }

    @Override
    public boolean handleResponse(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public boolean handleFault(MessageContext messageContext, Object endpoint) throws Exception {
        return true;
    }

    @Override
    public void afterCompletion(MessageContext messageContext, Object endpoint, Exception ex) throws Exception {
        UnitsBalance unitsBalance = getUnitsContext().getUnitsBalance();

        if (unitsBalance != null) {
            apiUnitsService.setUnitsHeaders(getHttpServletResponse());
        } else {
            logger.info("unitsBalance has been missed from ApiContext. Most likely that units storage is unavailable");
        }
    }

    private UnitsContext getUnitsContext() {
        return apiContextHolder.get().getUnitsContext();
    }

    // package private for unit testing
    ApiMethod getEndpointMethodMeta(Object endpoint) {
        return WsUtils.getEndpointMethodMeta(endpoint, ApiMethod.class);
    }

    // package private for unit testing
    HttpServletResponse getHttpServletResponse() {
        return WsUtils.getHttpServletResponse();
    }

}
