package ru.yandex.direct.api.v5.units.logging;

import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.springframework.stereotype.Component;

import ru.yandex.direct.api.v5.context.units.UnitsBucket;
import ru.yandex.direct.api.v5.context.units.UnitsLogData;
import ru.yandex.direct.core.units.api.UnitsBalance;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.api.v5.context.units.UnitsBucket.Type.OPERATOR;
import static ru.yandex.direct.api.v5.context.units.UnitsBucket.Type.OPERATOR_BRAND;
import static ru.yandex.direct.api.v5.context.units.UnitsBucket.Type.SUBCLIENT;
import static ru.yandex.direct.api.v5.context.units.UnitsBucket.Type.SUBCLIENT_BRAND;

@Component
@ParametersAreNonnullByDefault
class UnitsBucketFactory {

    private static final String UNEXPECTED_CLIENT_ID_MESSAGE = "Inconsistent units balance client id.";

    /**
     * Сформировать "корзину" баллов для записи в лог в случае успешного завершения обработки запроса в сервисе.
     * Структура заполняется с учётом роли держателя баллов на основе имеющихся данных в {@link UnitsLogData}.
     *
     * @param unitsLogData предзаполненная структура, описывающая контекст баллов в текущем запросе,
     *                     без {@link UnitsBucket} и {@code unitsUsedLogin}.
     * @param unitsBalance баланс держателя баллов.
     * @return Заполненная структура {@link UnitsBucket}.
     */
    UnitsBucket createUnitsHolderBucket(UnitsLogData unitsLogData, UnitsBalance unitsBalance) {
        Long bucketClientId = unitsBalance.getClientId();

        Long subclientBrandClientId = unitsLogData.getSubclientBrandClientId();
        Long subclientClientId = unitsLogData.getSubclientClientId();
        Long operatorBrandClientId = unitsLogData.getOperatorBrandClientId();
        Long operatorClientId = unitsLogData.getOperatorClientId();

        Set<Long> ids = Sets.newHashSet(subclientBrandClientId, subclientClientId, operatorBrandClientId,
                operatorClientId);

        checkState(ids.contains(bucketClientId), UNEXPECTED_CLIENT_ID_MESSAGE);

        final UnitsBucket.Type bucketType;
        final String bucketLogin;

        if (bucketClientId.equals(subclientBrandClientId)) {
            bucketType = SUBCLIENT_BRAND;
            bucketLogin = unitsLogData.getSubclientBrandLogin();
        } else if (bucketClientId.equals(subclientClientId)) {
            bucketType = SUBCLIENT;
            bucketLogin = unitsLogData.getSubclientClientLogin();
        } else if (bucketClientId.equals(operatorBrandClientId)) {
            bucketType = OPERATOR_BRAND;
            bucketLogin = unitsLogData.getOperatorBrandLogin();
        } else {
            bucketType = OPERATOR;
            bucketLogin = unitsLogData.getOperatorClientLogin();
        }

        return bucketFrom(unitsBalance)
                .withBucketType(bucketType)
                .withBucketLogin(bucketLogin);
    }

    /**
     * Сформировать "корзину" баллов для записи в лог в случае ошибки запроса.
     * Структура заполняется с учётом того, находится ли оператор под брендом
     * на основе имеющихся данных в {@link UnitsLogData}.
     *
     * @param unitsLogData         предзаполненная структура, описывающая контекст баллов в текущем запросе,
     *                             без {@link UnitsBucket} и {@code unitsUsedLogin}.
     * @param operatorUnitsBalance баланс баллов оператора.
     * @return Заполненная структура {@link UnitsBucket}.
     */
    UnitsBucket createOperatorBucket(UnitsLogData unitsLogData, UnitsBalance operatorUnitsBalance) {
        Long bucketClientId = operatorUnitsBalance.getClientId();

        Long operatorBrandClientId = unitsLogData.getOperatorBrandClientId();
        Long operatorClientId = unitsLogData.getOperatorClientId();

        checkState(bucketClientId.equals(operatorBrandClientId) || bucketClientId.equals(operatorClientId),
                UNEXPECTED_CLIENT_ID_MESSAGE);

        if (bucketClientId.equals(operatorBrandClientId)) {
            return bucketFrom(operatorUnitsBalance)
                    .withBucketType(OPERATOR_BRAND)
                    .withBucketLogin(unitsLogData.getOperatorBrandLogin());
        } else {
            return bucketFrom(operatorUnitsBalance)
                    .withBucketType(OPERATOR)
                    .withBucketLogin(unitsLogData.getOperatorClientLogin());
        }
    }

    private static UnitsBucket bucketFrom(UnitsBalance unitsBalance) {
        return new UnitsBucket()
                .withBucketClientId(unitsBalance.getClientId())
                .withBucketUnitsSpent(unitsBalance.spentInCurrentRequest())
                .withBucketUnitsBalance(unitsBalance.balance())
                .withBucketUnitsLimit(unitsBalance.getLimit());
    }

}
