package ru.yandex.direct.api.v5.validation;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.api.v5.ApiErrorCodes;
import ru.yandex.direct.api.v5.ApiFaultTranslations;
import ru.yandex.direct.core.ErrorCodes;
import ru.yandex.direct.core.entity.client.service.validation.ClientDefectTranslations;
import ru.yandex.direct.core.entity.grants.service.validation.GrantsDefectTranslations;
import ru.yandex.direct.core.validation.CommonDefectTranslations;
import ru.yandex.direct.currency.CurrencyAmount;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.types.DummyTranslatable;
import ru.yandex.direct.i18n.types.Identity;
import ru.yandex.direct.validation.result.Path;

import static ru.yandex.direct.api.v5.validation.DefectTypeCreators.wrongSelectionCriteria;
import static ru.yandex.direct.utils.StringUtils.joinLongsToString;
import static ru.yandex.direct.utils.StringUtils.joinToString;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@SuppressWarnings("WeakerAccess")
public class DefectTypes {
    public static final int MAX_STRING_SIZE_CODE = 5001;

    public static DefectType noRights() {
        return new DefectType(ErrorCodes.NO_RIGHTS,
                translations().noRightsShort(),
                (path, value) -> translations().noRightsDetailed(path));
    }

    public static DefectType noRightsCantWrite() {
        return new DefectType(ErrorCodes.NO_RIGHTS,
                translations().noRightsShort(),
                (path, value) -> translations().noRightsCantWrite());
    }

    public static DefectType notSupported() {
        return new DefectType(3500, translations().notSupportedShort());
    }

    public static DefectType restricted() {
        return new DefectType(3600, translations().restrictedShort());
    }

    public static DefectType maxIdsInSelection() {
        return wrongSelectionCriteria().withDetailedMessage(
                (path, value) -> translations().maxIdsInSelectionDetailed(path));
    }

    public static DefectType badParams() {
        return new DefectType(ApiErrorCodes.INVALID_REQUEST_PARAMS, translations().invalidRequestParamsShort());
    }

    public static DefectType badParamsNotLessThan(Number min) {
        return badParams().withDetailedMessage(
                // Просто вызываем toString чтобы переводить в том числе и BigDecimal
                (path, value) -> translations().notLessThan(path,
                        new DummyTranslatable(min.toString())));
    }

    public static DefectType badParamsNotLessThan(Number min, String field) {
        return badParams().withDetailedMessage(
                // Просто вызываем toString чтобы переводить в том числе и BigDecimal
                (path, value) -> translations().notLessThan(path(field(field)),
                        new DummyTranslatable(min.toString())));
    }

    public static DefectType badParamsNotGreaterThan(Number max) {
        return badParams().withDetailedMessage(
                // Просто вызываем toString чтобы переводить в том числе и BigDecimal
                (path, value) -> translations().notGreaterThan(path,
                        new DummyTranslatable(max.toString())));
    }

    public static DefectType badParamsInvalidFormat(String field) {
        return badParams().withDetailedMessage(
                (path, value) -> translations().invalidFormatWithPathDetailed(path(field(field))));
    }

    public static DefectType maxElementsInSelection(int max) {
        return wrongSelectionCriteria().withDetailedMessage(
                (path, value) -> translations().maxElementsInSelectionDetailed(path, new Identity(max)));
    }

    public static DefectType noNeededParamsInSelection() {
        return wrongSelectionCriteria().withDetailedMessage(
                translations().noNeededParamsInSelectionDetailed());
    }

    public static DefectType missedParamsInSelection(String missedParams) {
        return wrongSelectionCriteria().withDetailedMessage(
                translations().missedParamsInSelectionDetailed(missedParams));
    }

    public static DefectType incorrectPageNegativeOffset() {
        return new DefectType(4002,
                translations().incorrectPageShort(),
                translations().incorrectPageNegativeOffset());
    }

    public static DefectType incorrectPageOffsetExceeded(long border) {
        return incorrectPageNegativeOffset()
                .withDetailedMessage(translations().incorrectPageOffsetExceeded(new Identity(border)));
    }

    public static DefectType noOneOfRequiredParameters(String params) {
        return new DefectType(4003,
                translations().noOneOfRequiredParametersShort(),
                translations().noOneOfRequiredParametersDetailed(params));
    }

    public static DefectType mutuallyExclusiveParameters() {
        return new DefectType(4004,
                translations().mutuallyExclusiveParametersShort(),
                translations().mutuallyExclusiveParametersDetailed());
    }

    public static DefectType mutuallyExclusiveParameters(String params) {
        return mutuallyExclusiveParameters()
                .withDetailedMessage(translations().mutuallyExclusiveParametersWithElements(params));
    }

    public static DefectType mutuallyExclusiveParameters(Path first, Path second) {
        return mutuallyExclusiveParameters()
                .withDetailedMessage(translations().mutuallyExclusiveParametersTwoElements(first, second));
    }

    public static DefectType mutuallyExclusiveParameters(Path first, Path second, String secondValue) {
        return mutuallyExclusiveParameters()
                .withDetailedMessage(translations().mutuallyExclusiveParameterOneAndParameterTwoValue(first, second,
                        secondValue));
    }

    public static DefectType mixedTypes() {
        return new DefectType(4005,
                translations().mixedTypesShort(),
                m -> "");
    }

    public static DefectType mixedOperations() {
        return new DefectType(4006,
                translations().mixedOperationsShort(),
                translations().mixedOperationsShort());
    }

    public static DefectType incorrectPageNonPositiveLimit() {
        return incorrectPageNegativeOffset()
                .withDetailedMessage(translations().incorrectPageNonPositiveLimit());
    }

    public static DefectType incorrectPageLimitExceeded(long border) {
        return incorrectPageNegativeOffset()
                .withDetailedMessage(translations().incorrectPageLimitExceeded(new Identity(border)));
    }

    public static DefectType requiredButEmpty() { // ReqField в Перле
        return new DefectType(ApiErrorCodes.REQUIRED_FIELD,
                translations().requiredButEmptyShort(),
                (path, value) -> translations().requiredButEmptyDetailed(path));
    }

    public static DefectType requiredButEmptyTimeForGoalOrSegment() {
        return requiredButEmpty()
                .withDetailedMessage((path, value) -> translations().requiredTimeForGoalOrSegment(path));
    }

    public static DefectType absentValueInField() {
        return requiredButEmpty()
                .withDetailedMessage((p, v) -> translations().absentValueInFieldDetailed(p));
    }

    public static DefectType absentValueInField(String field) {
        return requiredButEmpty()
                .withDetailedMessage(translations().absentValueInFieldDetailed(path(field(field))));
    }

    public static DefectType maxStringSize(int max) {
        return new DefectType(MAX_STRING_SIZE_CODE,
                translations().maxStringSizeShort(),
                (path, value) -> translations().maxStringSizeDetailed(path, new Identity(max)));
    }

    public static DefectType lengthOfFieldValueMustNotExceed(int max) {
        return new DefectType(MAX_STRING_SIZE_CODE,
                translations().maxStringSizeShort(),
                (path, value) -> translations().lengthOfFieldValueMustNotExceed(path, Integer.toString(max)));
    }

    public static DefectType invalidChars() {
        return new DefectType(5002,
                translations().invalidCharsShort(),
                (path, value) -> translations().invalidCharsDetailed(path));
    }

    public static DefectType mustContainLetters() {
        return new DefectType(5002,
                translations().invalidCharsShort(),
                (path, value) -> translations().mustContainLettersDetailed(path));
    }

    public static DefectType mustContainLettersOrDigitsOrPunctuations() {
        return new DefectType(5002,
                translations().invalidCharsShort(),
                (path, value) -> translations().mustContainLettersOrDigitsOrPunctuationsDetailed(path));
    }

    public static DefectType invalidMinusMark() {
        return new DefectType(5002,
                translations().invalidCharsShort(),
                (path, value) -> translations().invalidMinusMarkDetailed(value.toString()));
    }

    public static DefectType emptyValue() {
        return new DefectType(5003,
                translations().emptyShort(),
                (path, value) -> translations().emptyDetailed(path));
    }

    public static DefectType emptyValue(String field) {
        return new DefectType(5003,
                translations().emptyShort(),
                (path, value) -> translations().emptyDetailed(path(field(field))));
    }

    public static DefectType invalidFormat() {
        return new DefectType(5004,
                translations().invalidFormatShort(),
                (path, value) -> {
                    Object possibleIdentity = convertToIdentityIfPossible(value);
                    return !path.isEmpty() ?
                            translations().invalidFormatWithPathDetailed(path) :
                            translations().invalidFormatDetailed(possibleIdentity);
                });
    }

    public static DefectType invalidFormat(String field) {
        return new DefectType(5004,
                translations().invalidFormatShort(),
                (path, value) -> translations().invalidFormatWithPathDetailed(path(field(field))));
    }

    public static DefectType invalidValue() {
        return new DefectType(ApiErrorCodes.INVALID_VALUE,
                translations().invalidValueShort(),
                (path, value) -> {
                    Object possibleIdentity = convertToIdentityIfPossible(value);
                    return !path.isEmpty() ?
                            translations().invalidValueWithPathDetailed(path, possibleIdentity) :
                            translations().invalidValueDetailed(possibleIdentity);
                });
    }

    public static DefectType invalidId() {
        return invalidValue().withDetailedMessage(
                (path, value) -> path.isEmpty() ?
                        translations().incorrectIdDetailed()
                        : translations().invalidValueShouldBePositiveDetailed(path)); // см. DIRECT-72525
    }

    public static DefectType invalidValueShouldBePositive() {
        return invalidValue().withDetailedMessage(
                (path, value) -> translations().invalidValueShouldBePositiveDetailed(path));
    }

    public static DefectType invalidValueNotLessThan(Money min) {
        return invalidValue().withDetailedMessage(
                (path, value) -> translations().notLessThan(path, CurrencyAmount.fromMoney(min)));
    }

    public static DefectType invalidValueNotLessThan(Number min) {
        return invalidValue().withDetailedMessage(
                // Просто вызываем toString чтобы переводить в том числе и BigDecimal
                (path, value) -> translations().notLessThan(path, new DummyTranslatable(min.toString())));
    }

    public static DefectType invalidValueNotGreaterThan(Money max) {
        return invalidValue().withDetailedMessage(
                (path, value) -> translations().notGreaterThan(path, CurrencyAmount.fromMoney(max)));
    }

    public static DefectType invalidValueShouldBeGreaterThan(Number min) {
        return invalidValue().withDetailedMessage(
                (path, value) -> translations().greaterThan(path, new Identity(min.longValue())));
    }

    public static DefectType invalidValueNotGreaterThan(Number max) {
        return invalidValue().withDetailedMessage(
                (path, value) -> translations().notGreaterThan(path, new Identity(max.longValue())));
    }

    public static DefectType unavailableCurrency(CurrencyCode currency, List<CurrencyCode> availableCurrencies) {
        return invalidValue().withDetailedMessage(
                ClientDefectTranslations.INSTANCE.invalidCurrencyDetails(
                        currency.name(),
                        availableCurrencies.stream()
                                .map(Enum::name)
                                .collect(Collectors.joining(","))));
    }

    public static DefectType inconsistentStateAllowEditCampaignAndAllowImportXls() {
        return inconsistentState()
                .withDetailedMessage(GrantsDefectTranslations.INSTANCE
                        .inconsistentStateAllowEditCampaignAndAllowImportXls());
    }

    public static DefectType invalidUseOfField() {
        return new DefectType(5006,
                translations().invalidUseOfFieldShort(),
                (path, value) -> translations().invalidUseOfFieldDetailed(path));
    }

    public static DefectType invalidUseOfFieldExtraFieldForGoalOrSegment() {
        return invalidUseOfField()
                .withDetailedMessage((path, value) -> translations().extraFieldForGoalOrSegment(path));
    }

    public static DefectType duplicatedElement() {
        return new DefectType(5007,
                translations().duplicatedElementShort(),
                (path, value) -> translations().duplicatedElementWithPathDetailed(path));
    }

    public static DefectType requiredAtLeastOneOfFields() {
        return new DefectType(5008,
                translations().requiredAtLeastOneOfFieldsShort(),
                (path, value) -> translations().requiredAtLeastOneOfFieldsDetailed(path.toString()));
    }

    public static DefectType requiredAtLeastOneOfFields(String fields) {
        return requiredAtLeastOneOfFields()
                .withDetailedMessage(translations().requiredAtLeastOneOfFieldsDetailed(fields));
    }

    public static DefectType requiredAtLeastOneOfFields(Path first, Path second) {
        return requiredAtLeastOneOfFields()
                .withDetailedMessage(translations().requiredAtLeastOneOfFieldsTwoElements(first, second));
    }

    public static DefectType possibleOnlyOneField() {
        return new DefectType(5009,
                translations().possibleOnlyOneFieldShort(),
                (path, value) -> translations().possibleOnlyOneFieldDetailed(path.toString()));
    }

    public static DefectType possibleOnlyOneField(String fields) {
        return possibleOnlyOneField()
                .withDetailedMessage(translations().possibleOnlyOneFieldDetailed(fields));
    }

    public static DefectType badGeo() {
        return new DefectType(5120,
                translations().badGeoShort(),
                translations().badGeoShort());
    }

    public static DefectType maxWords(int maxWords) {
        return new DefectType(5140,
                translations().maxWordsShort(),
                (path, value) -> translations().maxWordsDetailed(path, maxWords));
    }

    public static DefectType containsOnlyStopWords() {
        return new DefectType(5141,
                translations().stopWordsShort(),
                (path, value) -> translations().stopWordsDetailed(path));
    }

    public static DefectType keywordMaxLength(int maxLength) {
        return new DefectType(5142,
                translations().keywordMaxLengthShort(),
                translations().keywordMaxLengthDetailed(maxLength));
    }

    public static DefectType minusWords() {
        return new DefectType(5161,
                translations().minusWordsShort(),
                (path, value) -> translations().minusWordsOnlyDetailed(path));
    }

    public static DefectType minusWordMaxLength(int maxLength) {
        return new DefectType(5162,
                translations().minusWordMaxLengthShort(),
                (path, value) -> translations().minusWordMaxLengthOnlyDetailed(maxLength));
    }

    public static DefectType unsupportedAppStore() {
        return new DefectType(5180,
                translations().unsupportedAppStore());
    }

    public static DefectType inconsistentState() {
        return new DefectType(6000,
                translations().inconsistentStateShort(),
                translations().inconsistentStateDetailed());
    }

    public static DefectType inconsistentStateAlreadyExists() {
        return inconsistentState().withDetailedMessage(((path, value) -> translations().alreadyExistsDetailed(path)));
    }

    public static DefectType inconsistentStateTargetingCategoryUnavailable() {
        return inconsistentState()
                .withDetailedMessage(translations().inconsistentStateTargetingCategoryUnavailable());
    }

    public static DefectType inconsistentStateSitelinkSetCannotBeDeleted() {
        return inconsistentState().withDetailedMessage(translations().inconsistentStateSitelinkSetCannotBeDeleted());
    }

    public static DefectType notEligibleObject() {
        return new DefectType(6001,
                translations().notEligibleObjectShort(),
                (path, value) -> translations().notEligibleObjectDetailed(path));
    }

    public static DefectType inconsistentLanguageAndGeo() {
        return new DefectType(6101,
                translations().inconsistentLanguageAndGeoShort());
    }

    public static DefectType invalidCollectionSize() {
        return new DefectType(7000,
                translations().invalidCollectionSizeShort());
    }

    public static DefectType invalidCollectionSize(int min, int max) {
        return invalidCollectionSize().withDetailedMessage(
                (path, value) -> translations().invalidCollectionSizeDetailed(path,
                        new Identity(min), new Identity(max)));
    }

    public static DefectType minCollectionSize(int min) {
        return invalidCollectionSize(min, Integer.MAX_VALUE).withDetailedMessage(
                (path, value) -> translations().minCollectionSizeDetailed(path, new Identity(min)));
    }

    public static DefectType maxCollectionSize(int max) {
        return invalidCollectionSize(Integer.MIN_VALUE, max).withDetailedMessage(
                (path, value) -> translations().maxCollectionSizeDetailed(path, new Identity(max)));
    }

    public static DefectType maxCollectionSizeAdGroup(int max) {
        return maxCollectionSize(max)
                .withDetailedMessage((path, value) -> translations().adGroupLimitExceeded(new Identity(max)));
    }

    public static DefectType maxCollectionSizeCpmAdGroup(int max) {
        return maxCollectionSize(max)
                .withDetailedMessage((path, value) -> translations().cpmAdGroupLimitExceeded());
    }

    public static DefectType maxCollectionSizeAdGroupUserProfile(int max) {
        return maxCollectionSize(max)
                .withDetailedMessage((path, value) -> translations().adGroupUserProfileLimitExceeded());
    }

    public static DefectType maxElementsExceeded(int max) {
        return new DefectType(7001,
                translations().maxElementsExceededShort(),
                (path, value) -> translations().maxElementsExceededDetailed(new Identity(max)));
    }

    public static DefectType invalidRequest() {
        return new DefectType(8000, translations().invalidRequestShort());
    }

    public static DefectType absent() {
        return invalidRequest().withDetailedMessage((path, value) -> translations().absentDetailed(path));
    }

    public static DefectType emptyElementsList() {
        return absent().withDetailedMessage((path, value) -> translations().emptyElementsList(path));
    }

    public static DefectType notEmptyElementsList() {
        return absent().withDetailedMessage((path, value) -> translations().notEmptyElementsList(path));
    }

    public static DefectType absentIdsInSelection() {
        return absent().withDetailedMessage(translations().absentIdsInSelection());
    }

    public static DefectType absentElementInArray() {
        return absent().withDetailedMessage(translations().absentRequiredElement());
    }

    public static DefectType invalidRequestAgencyForbiddenFieldsOnly() {
        return invalidRequest().withDetailedMessage(translations().invalidRequestAgencyForbiddenFieldsOnlyDetailed());
    }

    public static DefectType invalidRequestOauthTokenRequired() {
        return invalidRequest().withDetailedMessage(translations().invalidRequestOauthTokenRequired());
    }

    public static DefectType invalidRequestRequiredRegularClient() {
        return invalidRequest().withDetailedMessage(translations().invalidRequestOauthTokenRequired());
    }


    public static DefectType badStatus() {
        return new DefectType(8300,
                translations().badStatusShort(),
                (path, value) -> translations().badStatusDetailed(path));
    }

    public static DefectType badStatusCampaignArchived() {
        return badStatus()
                .withDetailedMessage(translations().badCampaignStatusArchivedCanNotChange());
    }

    public static DefectType badStatusCampaignArchivedOnUpdateBids(Long campaignId) {
        return badStatus()
                .withDetailedMessage((path, value) -> translations()
                        .badCampaignStatusArchivedOnUpdateBids(new Identity(campaignId)));
    }

    public static DefectType badStatusCampaignArchivedOnAddRetargeting(Long campaignId) {
        return badStatus()
                .withDetailedMessage((path, value) -> translations()
                        .badCampaignStatusArchived(new Identity(campaignId)));
    }

    public static DefectType cantRemoveAudienceTargetFromArchivedCampaign(Long cid) {
        return new DefectType(8300,
                translations().badStatusShort(),
                (path, value) -> translations()
                        .cantRemoveAudienceTargetFromArchivedCampaignDetailed(new Identity(cid)));
    }

    public static DefectType unableToDelete() {
        return new DefectType(8301,
                translations().unableToDeleteShort(),
                (path, value) -> {
                    Object possibleIdentity = convertToIdentityIfPossible(value);
                    return !path.isEmpty() ?
                            translations().unableToDeleteWithPathDetailed(path, possibleIdentity) :
                            translations().unableToDeleteDetailed(possibleIdentity);
                });
    }

    public static DefectType unableToGetPin() {
        return new DefectType(8302,
                translations().unableToGetPin());
    }

    public static DefectType unableToArchiveCampaignWithMoney() {
        return new DefectType(8303,
                translations().cannotArchiveObject());
    }

    public static DefectType unableToArchiveNonStoppedCampaign() {
        return new DefectType(8303,
                translations().cannotArchiveObject());
    }

    public static DefectType cannotExtractObjectFromArchive() {
        return new DefectType(8304,
                translations().cannotExtractObjectFromArchive());
    }

    public static DefectType forbiddenToChange() {
        return badStatus()
                .withDetailedMessage(translations().forbiddenToChangeShort());
    }

    public static DefectType сantPerform() {
        return new DefectType(8305,
                translations().сantPerform());
    }

    public static DefectType notFound() {
        return new DefectType(8800,
                translations().notFoundShort(),
                (path, value) -> {
                    Object possibleIdentity = convertToIdentityIfPossible(value);
                    return !path.isEmpty() ?
                            translations().notFoundWithPathDetailed(path, possibleIdentity) :
                            translations().notFoundDetailed(possibleIdentity);
                });
    }

    public static DefectType notFound(String path) {
        return new DefectType(8800,
                translations().notFoundShort(),
                (pathIgnored, value) -> {
                    Object possibleIdentity = convertToIdentityIfPossible(value);
                    return translations().notFoundWithPathDetailed(path(field(path)), possibleIdentity);
                });
    }

    public static DefectType notFoundRetargeting() {
        return notFound().withDetailedMessage(translations().notFoundRetargetingDetailed());
    }

    public static DefectType notFoundAudienceTargetByParameters() {
        return notFound().withDetailedMessage(translations().notFoundAudienceTargetByParametersDetailed());
    }

    public static DefectType audienceTargetNotFound() {
        return notFound().withDetailedMessage(translations().audienceTargetNotFound());
    }

    public static DefectType maxElementsPerRequest(int max) {
        return new DefectType(9300,
                translations().maxElementsPerRequestShort(),
                (path, value) -> translations().maxElementsPerRequestDetailed(new Identity(max)));
    }

    public static DefectType maxElementsPerRequestAddRetargetings(int max) {
        return maxElementsPerRequest(max)
                .withDetailedMessage(translations().retargetingsAddLimitExceeded(new Identity(max)));
    }

    public static DefectType tooBroadSelectionClause(int max) {
        return new DefectType(9301,
                translations().tooBroadSelectionClauseShort(),
                (path, value) -> translations().tooBroadSelectionClauseDetailed(new Identity(max)));
    }

    public static DefectType duplicatedObject() {
        return new DefectType(ApiErrorCodes.DUPLICATED_OBJECT,
                translations().duplicatedObjectShort(),
                (path, value) -> !path.isEmpty() ?
                        translations().duplicatedObjectWithPathDetailed(path) :
                        translations().duplicatedObjectDetailed());
    }

    public static DefectType alreadyExists() {
        return new DefectType(9801,
                translations().alreadyExistsShort(),
                (path, value) -> translations().alreadyExistsDetailed(path));
    }

    public static DefectType duplicatedArrayItem() {
        return new DefectType(9802,
                translations().duplicatedArrayElementShort(),
                (path, value) -> translations().duplicatedArrayItemDetailed(path));
    }

    public static DefectType duplicatedElements(List<String> elements) {
        return new DefectType(9802, translations().duplicatedArrayElementShort(),
                (path, value) -> getDuplicatedElementsTranslatableDetailed(elements, path));
    }

    public static DefectType duplicatedElements(List<String> elements, String field) {
        return new DefectType(9802, translations().duplicatedArrayElementShort(),
                (path, value) -> getDuplicatedElementsTranslatableDetailed(elements, path(field(field))));
    }

    private static Translatable getDuplicatedElementsTranslatableDetailed(List<String> elements, Path path) {
        if (elements.size() > 1) {
            return getDuplicatedElementsPluralTranslatable(String.join(",", elements), path);
        } else if (elements.size() == 1) {
            return getDuplicatedElementsTranslatable(elements.get(0), path);
        } else {
            return getDuplicatedElementsTranslatable(path.toString(), path);
        }
    }

    private static Translatable getDuplicatedElementsTranslatable(String element, Path path) {
        if (path.getFieldName() != null) {
            return translations().duplicatedElementDetailedWithList(element, path.getFieldName());
        } else {
            return translations().duplicatedElementDetailedWithoutList(element);
        }
    }

    private static Translatable getDuplicatedElementsPluralTranslatable(String element, Path path) {
        if (path.getFieldName() != null) {
            return translations().duplicatedElementDetailedWithListPlural(element, path.getFieldName());
        } else {
            return translations().duplicatedElementDetailedWithoutListPlural(element);
        }
    }

    public static DefectType collectionMustContainElements(List<Long> elements) {
        return invalidValue().withDetailedMessage(
                (path, value) -> elements.size() > 1
                        ? translations().collectionMustContainElementsDetailedWithListPlural(
                        joinLongsToString(elements), path.getFieldName())
                        : translations().collectionMustContainElementsWithList(
                        elements.get(0).toString(), path.getFieldName()));
    }

    // WARNINGS
    public static DefectType warningDuplicated() {
        return new DefectType(10000,
                translations().duplicatedShort(),
                (path, value) -> translations().duplicatedDetailed(path));
    }

    public static DefectType warningDuplicatedRetargetingId() {
        return warningDuplicated().withDetailedMessage(translations().duplicatedRetargetingId());
    }

    public static DefectType warningDuplicatedCampaign() {
        return warningDuplicated().withDetailedMessage(translations().duplicatedCampaignId());
    }

    public static DefectType warningAlreadySuspended() {
        return new DefectType(10020,
                translations().alreadySuspendedShort(),
                (path, value) -> translations().alreadySuspendedDetailed(path));
    }

    public static DefectType warningAlreadySuspendedRetargeting() {
        return warningAlreadySuspended().withDetailedMessage(translations().alreadySuspendedRetargetingDetailed());
    }

    public static DefectType warningNotSuspended() {
        return new DefectType(10021,
                translations().notSuspendedShort(),
                (path, value) -> translations().notSuspendedDetailed(path));
    }

    public static DefectType warningNotSuspendedRetargeting() {
        return warningNotSuspended().withDetailedMessage(translations().notSuspendedRetargetingDetailed());
    }

    public static DefectType warningAlreadyArchived() {
        return new DefectType(10022,
                translations().objectAlreadyArchived());
    }

    public static DefectType warningNotArchived() {
        return new DefectType(10023, translations().objectNotArchived());
    }

    public static DefectType warningDuplicatedItem() {
        return new DefectType(10024,
                translations().duplicatedArrayElementShort(),
                (path, value) -> translations().duplicatedArrayItemDetailed(path));
    }

    public static DefectType warningAdExtensionIsDeleted() {
        return new DefectType(10025,
                translations().objectAlreadyMarkedAsDeletedShort(),
                (path, value) -> translations().objectAlreadyMarkedAsDeletedDetailed());
    }

    public static DefectType warningNoEffect() {
        return new DefectType(10030,
                translations().noEffect(),
                (p, v) -> null);
    }

    public static DefectType warningDuplicateSitelinkSet() {
        return new DefectType(10120,
                translations().duplicateSitelinkSet(),
                (p, v) -> null);
    }

    public static DefectType warningKeywordAlreadyExists() {
        return new DefectType(10140,
                translations().keywordAlreadyExistsShort(), (p, v) -> null);
    }

    public static DefectType dailyBudgetReset() {
        return new DefectType(10162,
                translations().dailyBudgetReset());
    }

    public static DefectType settingWillNotBeChanged() {
        return new DefectType(10163,
                translations().settingWillNotBeChanged());
    }

    public static DefectType parameterWillNotBeApplied() {
        return new DefectType(10165,
                translations().parameterWillNotBeApplied());
    }

    public static DefectType warningAdExtensionAlreadyExists() {
        return new DefectType(10164,
                translations().alreadyExistsShort());
    }

    public static DefectType warningObjectsAlreadyAssigned() {
        return new DefectType(10170,
                translations().objectsAlreadyBound());
    }

    public static DefectType warningObjectsAreNotAssigned() {
        return new DefectType(10171,
                translations().objectsAreNotBound());
    }

    public static DefectType unknownParameter() {
        return invalidRequest()
                .withDetailedMessage((p, v) -> ApiFaultTranslations.INSTANCE.detailedUnknownParameter(p.toString()));
    }

    public static DefectType hrefOrTurboPageIdMustBeNotNull() {
        return inconsistentState()
                .withDetailedMessage(
                        translations().oneOfHrefOrTurboLandingIdMustBeNotNull());
    }

    public static DefectType fieldMustBeInRange(Number min, Number max) {
        return invalidValue().withDetailedMessage(
                // Просто вызываем toString чтобы переводить в том числе и BigDecimal
                (path, value) -> translations().fieldMustBeInRangeTranslatable(path,
                        new DummyTranslatable(min.toString()),
                        new DummyTranslatable(max.toString())));
    }

    public static DefectType fieldShouldBePositive(String field) {
        return invalidValue().withDetailedMessage(
                (path, value) -> translations().fieldShouldBePositive(field));
    }

    public static DefectType disableAllAutotargetingCategoriesIsForbidden() {
        return invalidValue().withDetailedMessage(
                translations().disableAllAutotargetingCategoriesIsForbidden());
    }

    public static DefectType fieldMustBeInList(String field, Collection<?> allowedValues) {
        return invalidValue().withDetailedMessage(
                translations().fieldMustBeInList(field, joinToString(allowedValues)));
    }

    public static CommonDefectTranslations translations() {
        return CommonDefectTranslations.INSTANCE;
    }

    private static Object convertToIdentityIfPossible(Object obj) {
        if (obj instanceof Integer) {
            return new Identity((long) (int) obj);
        } else if (obj instanceof Long) {
            return new Identity((long) obj);
        } else {
            return obj;
        }
    }

}
