package ru.yandex.direct.api.v5.validation;

import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.validation.builder.Constraint;

public class SetBidsConstraints {
    private static final Logger logger = LoggerFactory.getLogger(SetBidsConstraints.class);

    /**
     * Проверка на размер набора элементов для изменения ставок для {@code AudienceTargets.setBids} или
     * {@code Bids.set}.
     * <p>
     * Создано в рамках DIRECT-70964
     *
     * @return экземпляр {@link Constraint<List<T>, DefectType>}, валидирующий на соответствие лимитам
     * @see <a href="https://st.yandex-team.ru/DIRECT-70964">DIRECT-70964</a>
     * <p>
     * Принимает на вход функции, получающие id по T, соответствующий лимит и функцию,
     * возвращающую {@link DefectType} по значению лимита для всего, для adGroup и для Campaign
     */
    public static <T> Constraint<List<T>, DefectType> bidsListSizeMaxLimit(
            Function<T, Long> getId, int idsLimit, Function<Integer, DefectType> idDefectTypeProvider,
            Function<T, Long> getAdGroupId, int adGroupIdsLimit, Function<Integer, DefectType> adGroupDefectTypeProvider,
            Function<T, Long> getCampaignId, int campaignIdsLimit, Function<Integer, DefectType> campaignDefectTypeProvider) {
        return bids -> {
            // ищем первый элемент, у которого заполнено хотя бы одно из полей в Selector'е
            Optional<T> bidOptional =
                    bids.stream()
                            // тут можно использовать #fieldCount(), но так нагляднее
                            .filter(s -> getId.apply(s) != null
                                    || getAdGroupId.apply(s) != null
                                    || getCampaignId.apply(s) != null)
                            .findFirst();
            if (!bidOptional.isPresent()) {
                logger.debug("There are no valid Selector in bidList. Skip limit check");
                return null;
            }

            T setBidItem = bidOptional.get();
            int length = bids.size();
            if (getId.apply(setBidItem) != null && length > idsLimit) {
                return idDefectTypeProvider.apply(idsLimit);
            } else if (getAdGroupId.apply(setBidItem) != null && length > adGroupIdsLimit) {
                return adGroupDefectTypeProvider.apply(adGroupIdsLimit);
            } else if (getCampaignId.apply(setBidItem) != null && length > campaignIdsLimit) {
                return campaignDefectTypeProvider.apply(campaignIdsLimit);
            }
            return null;
        };
    }
}


