package ru.yandex.direct.api.v5.validation

import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.ValidationResult
import java.util.function.Predicate
import kotlin.reflect.KProperty0
import kotlin.reflect.KProperty1

/*
 * Простые хелперы для удобного написания валидации на Kotlin (без ModelPropery)
 *
 * Пример использования:
 *
 * val result = validateObject(obj) {
 *     property(Obj::id) {
 *         check(CommonConstraints.notNull())
 *         check(NumberConstraints.lessThan(100))
 *         check(NumberConstraints.greaterThan(10))
 *     }
 *
 *     property(obj::name)
 *         .check(CommonConstraints.notNull())
 * }
 */

typealias D = DefectType

fun <T> validateObject(item: T, block: ItemValidationBuilder<T, D>.() -> Unit): ValidationResult<T, D> =
    ItemValidationBuilder.of<T, D>(item).apply(block).result

fun <T> validateObject(
    vr: ValidationResult<T, D>,
    block: ItemValidationBuilder<T, D>.() -> Unit
): ValidationResult<T, D> = ItemValidationBuilder(vr).apply(block).result

fun <T> validateList(list: List<T>?, block: ListValidationBuilder<T, D>.() -> Unit): ValidationResult<List<T>, D> =
    ListValidationBuilder.of<T, D>(list).apply(block).result

fun <T> validateList(
    vr: ValidationResult<List<T>, D>,
    block: ListValidationBuilder<T, D>.() -> Unit
): ValidationResult<List<T>, D> = ListValidationBuilder(vr).apply(block).result

// object attributes
fun <T, D, V> ItemValidationBuilder<T, D>.property(prop: KProperty1<T, V>): ItemValidationBuilder<V, D> =
    this.item(prop.get(this.result.value), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.property(prop: KProperty0<V>): ItemValidationBuilder<V, D> =
    this.item(prop.get(), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.property(
    prop: KProperty1<T, V>,
    init: ItemValidationBuilder<V, D>.() -> Unit
) = this.item(prop.get(this.result.value), prop.name).init()

fun <T, D, V> ItemValidationBuilder<T, D>.property(
    prop: KProperty1<T, V>,
    `when`: When<T, D>,
    init: ItemValidationBuilder<V, D>.() -> Unit,
) = this.item(prop.get(this.result.value), prop.name).init()

fun <T, D, V> ItemValidationBuilder<T, D>.property(
    prop: KProperty0<V>,
    init: ItemValidationBuilder<V, D>.() -> Unit
) = this.item(prop.get(), prop.name).init()

// for list attributes
fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(prop: KProperty1<T, List<V>?>): ListValidationBuilder<V, D> =
    this.list(prop.get(this.result.value), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(prop: KProperty0<List<V>?>): ListValidationBuilder<V, D> =
    this.list(prop.get(), prop.name)

fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(
    prop: KProperty1<T, List<V>?>,
    init: ListValidationBuilder<V, D>.() -> Unit
) = this.list(prop.get(this.result.value), prop.name).init()

fun <T, D, V> ItemValidationBuilder<T, D>.listProperty(
    prop: KProperty0<List<V>?>,
    init: ListValidationBuilder<V, D>.() -> Unit
) = this.list(prop.get(), prop.name).init()

// easier checks
fun <T, D> ItemValidationBuilder<T, D>.check(defect: D, predicate: Predicate<T>): ItemValidationBuilder<T, D> =
    this.check(Constraint.fromPredicate(predicate, defect))

fun <T, D> ItemValidationBuilder<T, D>.check(
    defect: D,
    `when`: When<T, D>,
    predicate: Predicate<T>
): ItemValidationBuilder<T, D> =
    this.check(Constraint.fromPredicate(predicate, defect), `when`)

fun <T, D> ListValidationBuilder<T, D>.checkEach(defect: D, predicate: Predicate<T>): ListValidationBuilder<T, D> =
    this.checkEach(Constraint.fromPredicate(predicate, defect))
