package ru.yandex.direct.api.v5.ws;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;
import com.fasterxml.jackson.databind.exc.UnrecognizedPropertyException;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.MethodParameter;
import org.springframework.stereotype.Component;
import org.springframework.ws.WebServiceMessage;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.server.endpoint.adapter.method.MethodArgumentResolver;
import org.springframework.ws.server.endpoint.adapter.method.MethodReturnValueHandler;

import ru.yandex.direct.api.v5.context.ApiContextHolder;
import ru.yandex.direct.api.v5.logging.ApiLogRecord;
import ru.yandex.direct.api.v5.ws.annotation.ApiRequest;
import ru.yandex.direct.api.v5.ws.annotation.ApiResponse;
import ru.yandex.direct.api.v5.ws.json.JsonMessage;
import ru.yandex.direct.api.v5.ws.validation.ApiObjectValidator;
import ru.yandex.direct.api.v5.ws.validation.IncorrectRequestApiException;
import ru.yandex.direct.api.v5.ws.validation.InvalidFormatApiException;
import ru.yandex.direct.api.v5.ws.validation.InvalidValueApiException;
import ru.yandex.direct.api.v5.ws.validation.JacksonUtil;
import ru.yandex.direct.api.v5.ws.validation.MissedParamsValueApiException;
import ru.yandex.direct.api.v5.ws.validation.UnknownParameterApiException;
import ru.yandex.direct.api.v5.ws.validation.WsdlValidatorFactory;
import ru.yandex.direct.core.TranslatableException;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_READER_BEAN_NAME;
import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME;
import static ru.yandex.direct.api.v5.ws.WsConstants.SOAP_MESSAGE_OBJECT_MAPPER_BEAN_NAME;


/**
 * Десериализует аргументы для метода эндпоинта и сериализует возвращаемое значение.
 * <p>
 * Следит за ошибками и выкидывает исключиение, наследника {@link TranslatableException}, с описанием ошибки,
 * в случае проблем десериализации или валидации получившегося аргумента.
 * <p>
 * Валидация аргумента здесь происходит по аннотациям JSR 303 на классе аргумента (генерируются по WSDL),
 * с помощью {@link ApiObjectValidator}.
 * <p>
 * Аргументы и метод в эндпоинте, должны быть помечены аннотациями {@link ApiRequest} и {@link ApiResponse}
 * <p>
 * Десериализация json и xml выполняется с помощью Jackson, для того, чтобы обрабатывать ошибки биндинга
 * (стандартная реализация JAXB эти ошибки игнорирует, полагаясь на проверку по схеме)
 */
@Component
public class ApiObjectsArgumentAndReturnValueResolver implements MethodArgumentResolver, MethodReturnValueHandler {
    private static final Logger logger = LoggerFactory.getLogger(ApiObjectsArgumentAndReturnValueResolver.class);

    private final ConcurrentMap<Class<?>, JAXBContext> jaxbContexts = new ConcurrentHashMap<>();

    private final ObjectMapper jsonMessageObjectReader;
    private final ObjectMapper jsonMessageObjectWriter;
    private final XmlMapper soapMessageObjectMapper;
    private final ApiObjectValidator apiObjectValidator;
    private final ApiContextHolder apiContextHolder;
    private final WsdlValidatorFactory wsdlValidatorFactory;


    @Autowired
    public ApiObjectsArgumentAndReturnValueResolver(
            @Qualifier(JSON_MESSAGE_OBJECT_READER_BEAN_NAME) ObjectMapper jsonMessageObjectReader,
            @Qualifier(JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME) ObjectMapper jsonMessageObjectWriter,
            @Qualifier(SOAP_MESSAGE_OBJECT_MAPPER_BEAN_NAME) XmlMapper soapMessageObjectMapper,
            ApiObjectValidator apiObjectValidator,
            ApiContextHolder apiContextHolder,
            WsdlValidatorFactory wsdlValidatorFactory) {
        this.jsonMessageObjectReader = jsonMessageObjectReader;
        this.jsonMessageObjectWriter = jsonMessageObjectWriter;
        this.apiObjectValidator = apiObjectValidator;
        this.soapMessageObjectMapper = soapMessageObjectMapper;
        this.apiContextHolder = apiContextHolder;
        this.wsdlValidatorFactory = wsdlValidatorFactory;
    }

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        Class<?> parameterType = parameter.getParameterType();
        return (parameter.getParameterAnnotation(ApiRequest.class) != null)
                && parameterType.isAnnotationPresent(XmlRootElement.class);
    }

    @Override
    public Object resolveArgument(MessageContext messageContext, MethodParameter parameter) throws Exception {
        ApiMessage request = (ApiMessage) messageContext.getRequest();
        Object apiRequestPayload;

        try {
            if (request instanceof JsonMessage) {
                JsonMessage jsonMessage = (JsonMessage) request;

                setApiLogRecordParams(jsonMessage.getPayloadSource().getObject());
                wsdlValidatorFactory.getValidator(jsonMessage.getService()).validateJsonRequest(jsonMessage);

                apiRequestPayload = deserializeJsonObject(jsonMessage, parameter.getParameterType());
            } else {
                apiRequestPayload = deserializeXmlObject(request, parameter.getParameterType());
                setApiLogRecordParams(apiRequestPayload);
            }
        } catch (UnrecognizedPropertyException ex) {
            logger.warn("resolveArgument error:", ex);
            throw UnknownParameterApiException.fromJacksonUnrecognizedPropertyException(ex);
        } catch (InvalidFormatException ex) {
            logger.warn("resolveArgument error:", ex);
            throw InvalidFormatApiException.fromJacksonInvalidFormatException(ex);
        } catch (JsonMappingException ex) {
            logger.warn("resolveArgument error:", ex);
            String exceptionPath = JacksonUtil.getMappingExceptionPath(ex);
            if (isBlank(exceptionPath)) {
                throw new IncorrectRequestApiException();
            }
            throw InvalidValueApiException.fromJsonMappingException(ex, exceptionPath);
        }
        if (apiRequestPayload == null) {
            throw new MissedParamsValueApiException();
        }
        request.setApiRequestPayload(apiRequestPayload);
        apiObjectValidator.validate(apiRequestPayload);

        return apiRequestPayload;
    }

    private <T> T deserializeJsonObject(JsonMessage request, Class<T> objectType) throws JsonMappingException {
        logger.debug("deserialize Json");
        Object payload = request.getPayloadSource().getObject();
        try (TraceProfile profile = Trace.current().profile("api:json:convertRequest")) {
            return jsonMessageObjectReader.convertValue(payload, objectType);
        } catch (IllegalArgumentException ex) {
            // При конвертации mapper.convertValue исключение оборачивается в IllegalArgumentException
            // нужно достать оригинальную ошибку биндинга
            if (ex.getCause() instanceof JsonMappingException) {
                throw (JsonMappingException) ex.getCause();
            } else {
                throw ex;
            }
        }
    }

    private <T> T deserializeXmlObject(WebServiceMessage request, Class<T> objectType)
            throws XMLStreamException, java.io.IOException {
        logger.debug("deserialize Xml");
        try (TraceProfile profile = Trace.current().profile("api:soap:unmarshalRequest")) {
            XMLStreamReader xmlStreamReader =
                    XMLInputFactory.newInstance().createXMLStreamReader(request.getPayloadSource());
            return soapMessageObjectMapper.readValue(xmlStreamReader, objectType);
        }
    }

    private void setApiLogRecordParams(Object params) {
        try {
            ApiLogRecord apiLogRecord = apiContextHolder.get().getApiLogRecord();
            String paramsString = jsonMessageObjectWriter.writeValueAsString(params);

            apiLogRecord.withParams(paramsString);
        } catch (JsonProcessingException e) {
            logger.error("Can't serialize params", e);
        }
    }

    @Override
    public boolean supportsReturnType(MethodParameter returnType) {
        Class<?> parameterType = returnType.getParameterType();
        return (returnType.getMethodAnnotation(ApiResponse.class) != null)
                && parameterType.isAnnotationPresent(XmlRootElement.class);
    }

    @Override
    public void handleReturnValue(MessageContext messageContext, MethodParameter returnType, Object returnValue)
            throws Exception {
        ApiMessage response = (ApiMessage) messageContext.getResponse();
        response.setApiResponsePayload(returnValue);

        if (response instanceof JsonMessage) {
            // json message
            ((JsonMessage) messageContext.getResponse()).getPayloadResult().setObject(returnValue);
        } else {
            // soap or xml message
            JAXBContext jaxbContext = getJaxbContext(returnType.getParameterType());
            Marshaller marshaller = jaxbContext.createMarshaller();
            marshaller.marshal(returnValue, response.getPayloadResult());
        }
    }

    private JAXBContext getJaxbContext(Class<?> clazz) throws JAXBException {
        JAXBContext jaxbContext = jaxbContexts.get(clazz);
        if (jaxbContext == null) {
            jaxbContext = JAXBContext.newInstance(clazz);
            jaxbContexts.putIfAbsent(clazz, jaxbContext);
        }
        return jaxbContext;
    }
}
