package ru.yandex.direct.api.v5.ws;

import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.springframework.beans.factory.BeanFactoryUtils;
import org.springframework.context.ApplicationContext;
import org.springframework.context.i18n.LocaleContext;
import org.springframework.context.i18n.SimpleLocaleContext;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.ws.transport.http.HttpTransportConstants;
import org.springframework.ws.transport.http.MessageDispatcherServlet;
import org.springframework.ws.wsdl.WsdlDefinition;

import static ru.yandex.direct.common.util.HttpUtil.getRequestUrl;

/**
 * We need to replace default MessageDispatcherServlet to
 * 1. support custom locale resolving mechanism
 * 2. get wsdl by "?wsdl" instead of ".wsdl"
 */
public class ApiWsDispatcherServlet extends MessageDispatcherServlet {
    private static final String WSDL_SUFFIX_NAME = "?wsdl";
    private LocaleResolver localeResolver;
    private Map<String, WsdlDefinition> wsdlDefinitions;

    @Override
    protected void initStrategies(ApplicationContext context) {
        super.initStrategies(context);
        initLocaleContextResolver(context);
        initWsdlDefinitions(context);
    }

    private void initLocaleContextResolver(ApplicationContext context) {
        // required bean; this line throws NoSuchBeanDefinitionException if suitable bean is not found
        localeResolver = context.getBean(LocaleResolver.class);
    }

    /**
     * Partially duplicates method from supertype because wsdlDefinitions of supertype is private
     *
     * @param context instance of ApplicationContext
     */
    private void initWsdlDefinitions(ApplicationContext context) {
        wsdlDefinitions = BeanFactoryUtils
                .beansOfTypeIncludingAncestors(context, WsdlDefinition.class, true, false);
    }

    @Override
    protected LocaleContext buildLocaleContext(final HttpServletRequest request) {
        return new SimpleLocaleContext(localeResolver.resolveLocale(request));
    }

    /**
     * Determines the {@link WsdlDefinition} for a given request, or {@code null} if none is found.
     *
     * <p>Default implementation checks whether the request method is {@code GET}, whether the request uri ends with
     * {@code "?wsdl"}, and if there is a {@code WsdlDefinition} with the same name as the filename in the
     * request uri.
     *
     * @param request the {@code HttpServletRequest}
     * @return a definition, or {@code null}
     */
    @Override
    protected WsdlDefinition getWsdlDefinition(HttpServletRequest request) {
        String requestUrl = getRequestUrl(request);
        if (HttpTransportConstants.METHOD_GET.equals(request.getMethod()) &&
                requestUrl.endsWith(WSDL_SUFFIX_NAME)) {
            return wsdlDefinitions.get(extractWsdlName(requestUrl));
        } else {
            return null;
        }
    }

    /**
     * Extracts wsdl name from URL ends with WSDL_SUFFIX_NAME.
     *
     * @param wsdlUrl url to wsdl
     * @return wsdl name
     */
    private String extractWsdlName(String wsdlUrl) {
        int end = wsdlUrl.length() - WSDL_SUFFIX_NAME.length();
        int begin = wsdlUrl.lastIndexOf("/", end) + 1;
        return wsdlUrl.substring(begin, end);
    }
}
