package ru.yandex.direct.api.v5.ws;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.web.servlet.ModelAndView;
import org.springframework.ws.InvalidXmlException;
import org.springframework.ws.NoEndpointFoundException;
import org.springframework.ws.WebServiceException;
import org.springframework.ws.WebServiceMessage;
import org.springframework.ws.WebServiceMessageException;
import org.springframework.ws.WebServiceMessageFactory;
import org.springframework.ws.transport.WebServiceConnection;
import org.springframework.ws.transport.WebServiceMessageReceiver;
import org.springframework.ws.transport.http.HttpTransportConstants;
import org.springframework.ws.transport.http.WebServiceMessageReceiverHandlerAdapter;

import ru.yandex.direct.api.v5.units.ApiUnitsService;
import ru.yandex.direct.api.v5.ws.exceptionresolver.ApiExceptionResolver;


/**
 * Отправляет сообщения к {@link WebServiceMessageReceiver}.
 * Перехватывает ошибки создания сообщений, парсинга xml/json и возвращает ответ, транслируя ошибки с помощью
 * {@link ApiExceptionResolver}
 */
public class ApiWsHandlerAdapter extends WebServiceMessageReceiverHandlerAdapter {

    private final ApiExceptionResolver apiExceptionResolver;
    private final ApiUnitsService apiUnitsService;

    public ApiWsHandlerAdapter(WebServiceMessageFactory messageFactory, ApiExceptionResolver apiExceptionResolver,
                               ApiUnitsService apiUnitsService) {
        setMessageFactory(messageFactory);
        this.apiExceptionResolver = apiExceptionResolver;
        this.apiUnitsService = apiUnitsService;
    }

    void handlePostMethod(
            HttpServletRequest httpServletRequest,
            HttpServletResponse httpServletResponse,
            Object handler) throws Exception {
        WebServiceConnection
                connection = new ApiHttpServletConnection(httpServletRequest, httpServletResponse);
        try {
            handleConnection(connection, (WebServiceMessageReceiver) handler);
        } catch (InvalidXmlException | WebServiceMessageException ex) {
            handleMessageCreationException(httpServletResponse, ex);
        } catch (Exception ex) {
            if (ex instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            handleException(httpServletResponse, ex);
        }
    }

    @Override
    public ModelAndView handle(
            HttpServletRequest httpServletRequest,
            HttpServletResponse httpServletResponse,
            Object handler) throws Exception {
        if (HttpTransportConstants.METHOD_POST.equals(httpServletRequest.getMethod())) {
            handlePostMethod(httpServletRequest, httpServletResponse, handler);
        } else {
            handleNonPostMethod(httpServletRequest, httpServletResponse, handler);
        }
        return null;
    }

    private void handleMessageCreationException(
            HttpServletResponse response, WebServiceException ex) throws IOException {
        handleException(response, new MessageCreationApiException(ex));
    }

    @Override
    protected void handleNoEndpointFoundException(NoEndpointFoundException ex,
                                                  WebServiceConnection connection, WebServiceMessageReceiver receiver) throws IOException {
        // Пробрасываем наше исключение наверх чтобы оно было обработано единообразно данным классом
        throw new NoEndpointFoundApiException(ex);
    }

    private void handleException(
            HttpServletResponse response, Exception ex) throws IOException {
        WebServiceMessage responseMessage = getMessageFactory().createWebServiceMessage();

        // Мы ожидаем, что какой-то fault-message будет помещен в responseMessage. Иначе надо 500-ить,
        // так как в текущем виде responseMessage не пригоден для возвращения клиенту. Т.е. просто
        // промолчать нельзя
        if (!apiExceptionResolver.resolveException(responseMessage, ex)) {
            throw new AssertionError("Should not happen");
        }

        apiUnitsService.setUnitsHeaders(response);

        responseMessage.writeTo(response.getOutputStream());
    }
}
