package ru.yandex.direct.api.v5.ws;

import java.util.Optional;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerAdapter;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.ws.transport.http.MessageDispatcherServlet;
import org.springframework.ws.transport.http.WebServiceMessageReceiverHandlerAdapter;


/**
 * Компонент для spring-ws, который делегирует принятие запроса к другим HandlerAdapter'ам, в зависимости от URI
 */
@Component(MessageDispatcherServlet.DEFAULT_MESSAGE_RECEIVER_HANDLER_ADAPTER_BEAN_NAME)
public class DelegatingWebServiceMessageReceiverHandlerAdapter extends WebServiceMessageReceiverHandlerAdapter {

    private final HandlerAdapterDelegateRule[] delegateEntries;

    @Autowired
    public DelegatingWebServiceMessageReceiverHandlerAdapter(HandlerAdapterDelegateRule... delegateEntries) {
        this.delegateEntries = delegateEntries;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        // Переопределяем, чтобы выключить проверку обязательности фабрики. В этом адаптере она не нужна
    }

    @Override
    public ModelAndView handle(
            HttpServletRequest httpServletRequest,
            HttpServletResponse httpServletResponse, Object handler) throws Exception {
        Optional<HandlerAdapter> handlerAdapter = getDelegateFor(httpServletRequest.getRequestURI());
        if (!handlerAdapter.isPresent()) {
            httpServletResponse.setStatus(HttpServletResponse.SC_NOT_FOUND);
            return null;
        }

        try {
            return handlerAdapter.get().handle(httpServletRequest, httpServletResponse, handler);
        } catch (Exception ex) {
            logger.error("Unexpected exception", ex);
            httpServletResponse.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            return null;
        }
    }

    private Optional<HandlerAdapter> getDelegateFor(String uri) {
        // подразумевается, что кол-во делегатов совсем небольшое, поэтому поиск просто перебором
        for (HandlerAdapterDelegateRule delegateEntry : delegateEntries) {
            if (delegateEntry.isSuitableFor(uri)) {
                return Optional.of(delegateEntry.getHandlerAdapter());
            }
        }
        return Optional.empty();
    }
}
