package ru.yandex.direct.api.v5.ws.exceptionresolver;

import java.util.LinkedHashMap;
import java.util.Map;

import javax.xml.namespace.QName;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import org.springframework.stereotype.Component;
import org.springframework.ws.WebServiceMessage;

import ru.yandex.direct.api.v5.ws.json.JsonMessage;


/**
 * Creates error response at {@link JsonMessage}
 *
 * @see JsonMessage
 * @see ApiExceptionResolver
 */
@Component
public class JsonFaultResponseCreator implements FaultResponseCreator {
    private final ObjectMapper objectMapper;

    public JsonFaultResponseCreator() {
        // it need to convert field names of fault info bean to underscores to support
        // an existent json fault message format
        objectMapper = new ObjectMapper()
                .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
    }

    @Override
    public boolean support(WebServiceMessage response) {
        return response instanceof JsonMessage;
    }

    @Override
    @SuppressWarnings("unchecked")
    public void createFaultResponse(WebServiceMessage response, String faultMessage, QName webFaultQName,
                                    Object faultInfo) {
        Map<String, Object> faultDetail;
        if (faultInfo != null) {
            faultDetail = objectMapper.convertValue(faultInfo, Map.class);
        } else {
            // if the fault info bean is absent, we need to create fault anyway because
            // default spring-ws exception resolvers don't know anything about JsonMessage
            faultDetail = new LinkedHashMap<>();
        }
        // at the json fault response, faultMessage is placed at the same level as other info from faultInfo bean
        // and must be called 'error_string'
        faultDetail.put("error_string", faultMessage);
        ((JsonMessage) response).setApiFault(faultInfo);
        ((JsonMessage) response).setFaultDetail(faultDetail);
    }
}
