package ru.yandex.direct.api.v5.ws.json;

import java.io.IOException;
import java.io.OutputStream;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.ws.WebServiceMessage;
import org.springframework.ws.transport.TransportOutputStream;

import ru.yandex.direct.api.v5.ws.ApiMessage;
import ru.yandex.direct.tracing.Trace;

import static ru.yandex.direct.api.v5.ws.WsConstants.HEADER_CONTENT_TYPE;
import static ru.yandex.direct.api.v5.ws.WsConstants.HEADER_REQUEST_ID;


/**
 * Represents JSON PoWeReD {@link WebServiceMessage}
 */
public class JsonMessage extends ApiMessage {
    public static final String JSON_CONTENT_TYPE = "application/json; charset=utf-8";

    private ObjectMapper objectMapperWriter;

    private Object payload;
    private Object faultDetail;

    /**
     * @param objectMapperWriter jackson {@link ObjectMapper} to serialize content of message to stream
     * @param service            service name
     * @param operation          operation
     * @param payload            payload of message
     */
    public JsonMessage(ObjectMapper objectMapperWriter, String service, String operation, Object payload) {
        this.objectMapperWriter = objectMapperWriter;
        this.service = service;
        this.operation = operation;
        this.payload = payload;
    }

    public JsonMessage(ObjectMapper objectMapperWriter, String service) {
        this(objectMapperWriter, service, null, null);
    }

    @Override
    public ObjectSourceAndResult getPayloadSource() {
        return new ObjectSourceAndResult(() -> this.payload, obj -> payload = obj);
    }

    @Override
    public ObjectSourceAndResult getPayloadResult() {
        return new ObjectSourceAndResult(() -> this.payload, obj -> payload = obj);
    }

    @Override
    public void writeTo(OutputStream outputStream) throws IOException {
        if (outputStream instanceof TransportOutputStream) {
            TransportOutputStream transportOutputStream = (TransportOutputStream) outputStream;
            transportOutputStream.addHeader(HEADER_CONTENT_TYPE, JSON_CONTENT_TYPE);
            transportOutputStream.addHeader(HEADER_REQUEST_ID, String.valueOf(Trace.current().getSpanId()));
        }

        if (!hasFault()) {
            objectMapperWriter.writeValue(outputStream, new JsonResponseEnvelope().withResult(payload));
        } else {
            objectMapperWriter.writeValue(outputStream, new JsonErrorEnvelope().withError(faultDetail));
        }
    }

    public JsonMessage withService(final String service) {
        this.service = service;
        return this;
    }

    public JsonMessage withOperation(final String operation) {
        this.operation = operation;
        return this;
    }

    public Object getFaultDetail() {
        return faultDetail;
    }

    public void setFaultDetail(Object error) {
        this.faultDetail = error;
    }

    public boolean hasFault() {
        return faultDetail != null;
    }
}
