package ru.yandex.direct.api.v5.ws.json;

import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.AntPathMatcher;
import org.springframework.ws.WebServiceMessageFactory;

import ru.yandex.direct.api.v5.ws.RequestUriProvider;
import ru.yandex.direct.api.v5.ws.WebServiceMessageCreationException;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_READER_BEAN_NAME;
import static ru.yandex.direct.api.v5.ws.WsConstants.JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME;


@Component
public class JsonMessageFactory implements WebServiceMessageFactory {

    private static final String SERVICE_VARIABLE = "service";
    private static final String URL_PATTERN = "json/v5/{" + SERVICE_VARIABLE + "}";

    private final ObjectMapper jsonMessageObjectReader;
    private final ObjectMapper jsonMessageObjectWriter;

    private final RequestUriProvider requestUriProvider;

    private final AntPathMatcher pathMatcher;

    @Autowired
    public JsonMessageFactory(
            @Qualifier(JSON_MESSAGE_OBJECT_READER_BEAN_NAME) ObjectMapper jsonMessageObjectReader,
            @Qualifier(JSON_MESSAGE_OBJECT_WRITER_BEAN_NAME) ObjectMapper jsonMessageObjectWriter,
            RequestUriProvider requestUriProvider) {
        this.jsonMessageObjectWriter = jsonMessageObjectWriter;
        pathMatcher = new AntPathMatcher();
        pathMatcher.setCachePatterns(true);
        this.jsonMessageObjectReader = jsonMessageObjectReader;
        this.requestUriProvider = requestUriProvider;
    }

    @Override
    public JsonMessage createWebServiceMessage() {
        return new JsonMessage(jsonMessageObjectWriter, getService());
    }

    @Override
    public JsonMessage createWebServiceMessage(InputStream inputStream) {
        JsonRequestEnvelope jsonRequestEnvelope;
        try (TraceProfile profile = Trace.current().profile("api:json:parseEnvelope")) {
            jsonRequestEnvelope = jsonMessageObjectReader.readValue(inputStream, JsonRequestEnvelope.class);
        } catch (IOException e) {
            throw new WebServiceMessageCreationException("can deserialize json request", e);
        }
        return new JsonMessage(jsonMessageObjectWriter, getService(),
                jsonRequestEnvelope.getMethod(), jsonRequestEnvelope.getParams());
    }

    @Nullable
    private String getService() {
        Optional<String> requestUri = requestUriProvider.getRequestUri();
        if (requestUri.isPresent()) {
            String requestUriStr = StringUtils.strip(requestUri.get(), "/");
            return pathMatcher.extractUriTemplateVariables(URL_PATTERN, requestUriStr).get(SERVICE_VARIABLE);
        }
        return null;
    }
}
