package ru.yandex.direct.api.v5.ws.validation;

import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.type.TypeFactory;

import ru.yandex.direct.api.v5.ApiFaultTranslations;
import ru.yandex.direct.i18n.Translatable;

/**
 * Ошибка, возникающая при невозможности привести значение из документа запроса к типу в объекте запроса
 */
public class InvalidValueApiException extends IncorrectRequestApiException {
    // см. тексты com.fasterxml.jackson.databind.DeserializationContext.handleUnexpectedToken
    private static final Pattern CANT_DESERIALIZE_MESSAGE_PATTERN =
            Pattern.compile("^Can ?not deserialize instance of `?(\\S+?)`? out of (\\S+) token");
    private static final String START_ARRAY_TOKEN = "START_ARRAY";
    private static final Pattern CANT_COERCE_FLOATING_POINT_PATTERN =
            Pattern.compile("^Can not coerce a floating-point value \\([^)]+\\) into Long;");

    public InvalidValueApiException(String path) {
        super(ApiFaultTranslations.INSTANCE.detailedIncorrectValue(path));
    }

    public InvalidValueApiException(Throwable cause, Translatable detailedMessage) {
        super(cause, detailedMessage);
    }

    public static InvalidValueApiException fromJsonMappingException(JsonMappingException ex, String exceptionPath) {
        MessageParseResult parseResult = parseJsonMappingException(ex);

        if (parseResult.isClassAssignableFromTargetClass(List.class)) {
            // Ожидается массив, а приходит что-то другое
            return new InvalidValueApiException(
                    ex, ApiFaultTranslations.INSTANCE.detailedIncorrectValueFieldShouldBeArray(exceptionPath));
        }

        if (parseResult.isDirectlyUnderArray()
                && parseResult.isClassAssignableFromTargetClass(Enum.class)) {
            // Ожидается Enum
            assert parseResult.getTargetClass().isPresent();
            return new InvalidValueApiException(
                    ex, ApiFaultTranslations.INSTANCE.detailedInvalidFormatExpectedEnumInArray(
                    exceptionPath, XmlBindUtil.getEnumXmlElements(parseResult.getTargetClass().get())));
        }

        if (parseResult.getToken().map(START_ARRAY_TOKEN::equals).orElse(false)) {
            // Ожидается не массив, а приходит массив
            // Чтобы не путать пользователя сообщаем о том, что ожидаем массив только если находимся не в массиве
            if (!parseResult.isDirectlyUnderArray()) {
                return new InvalidValueApiException(
                        ex, ApiFaultTranslations.INSTANCE.detailedIncorrectValueFieldShouldNotBeArray(exceptionPath));
            }
            // Попытка запихнуть массив в элемент массива целых чисел
            if (parseResult.isClassAssignableFromTargetClass(Long.class)
                    || parseResult.isClassAssignableFromTargetClass(Integer.class)) {
                return new InvalidValueApiException(ex,
                        ApiFaultTranslations.INSTANCE.detailedInvalidFormatExpectedIntegerInArray(exceptionPath));
            }
        }

        if (parseResult.isClassAssignableFromTargetClass(Long.class)) {
            if (parseResult.isDirectlyUnderArray()) {
                return new InvalidValueApiException(ex,
                        ApiFaultTranslations.INSTANCE.detailedInvalidFormatExpectedIntegerInArray(exceptionPath));
            } else {
                return new InvalidValueApiException(ex,
                        ApiFaultTranslations.INSTANCE.detailedInvalidFormatExpectedInteger(exceptionPath));
            }

        }

        if (parseResult.isClassAssignableFromTargetClass(String.class)) {
            if (parseResult.isDirectlyUnderArray()) {
                return new InvalidValueApiException(ex,
                        ApiFaultTranslations.INSTANCE.detailedInvalidFormatExpectedStringInArray(exceptionPath));
            } else {
                return new InvalidValueApiException(ex,
                        ApiFaultTranslations.INSTANCE.detailedInvalidFormatExpectedString(exceptionPath));
            }

        }
        return new InvalidValueApiException(ex, ApiFaultTranslations.INSTANCE.detailedIncorrectValue(exceptionPath));
    }

    private static MessageParseResult parseJsonMappingException(JsonMappingException ex) {
        // Другого способа определить класс, на котором возникла ошибка не нашлось
        // Интеграция с jackson проверяется тестами в ApiObjectsArgumentAndReturnValueResolverJacksonIntegrationTest
        String message = ex.getMessage();
        PathWithArrayPrefix pathWithArrayPrefix = PathWithArrayPrefix.fromPath(ex.getPath());
        boolean directlyUnderArray = pathWithArrayPrefix.getInArrayPath().isEmpty();

        Matcher matcher = CANT_DESERIALIZE_MESSAGE_PATTERN.matcher(message);
        if (!matcher.find()) {
            if (CANT_COERCE_FLOATING_POINT_PATTERN.matcher(message).find()) {
                return new MessageParseResult(Long.class, null, directlyUnderArray);
            } else {
                return new MessageParseResult(null, null, directlyUnderArray);
            }
        }

        String className = matcher.group(1);
        String token = matcher.group(2);
        Class<?> cls = null;
        try {
            cls = TypeFactory.defaultInstance().findClass(className);
        } catch (ClassNotFoundException ignore) {
        }
        return new MessageParseResult(cls, token, directlyUnderArray);
    }

    private static class MessageParseResult {
        private final Class<?> targetClass;
        private final String token;
        private final boolean directlyUnderArray;

        MessageParseResult(@Nullable Class<?> targetClass, @Nullable String token, boolean directlyUnderArray) {
            this.targetClass = targetClass;
            this.token = token;
            this.directlyUnderArray = directlyUnderArray;
        }

        public Optional<Class<?>> getTargetClass() {
            return Optional.ofNullable(targetClass);
        }

        public Optional<String> getToken() {
            return Optional.ofNullable(token);
        }

        public boolean isDirectlyUnderArray() {
            return directlyUnderArray;
        }

        public boolean isClassAssignableFromTargetClass(Class<?> clz) {
            return getTargetClass().map(clz::isAssignableFrom).orElse(false);
        }
    }
}
