package ru.yandex.direct.api.v5.ws.validation;

import javax.validation.ConstraintViolation;
import javax.validation.Path;

import com.google.common.collect.Iterables;

import ru.yandex.direct.api.v5.ApiFaultTranslations;
import ru.yandex.direct.i18n.Translatable;

import static ru.yandex.direct.api.v5.ws.validation.ValidationUtil.simpleViolationPathConverter;

/**
 * Ошибка валидции объекта запроса, при существовании ограничения NotNull на поле бина
 */
class NullValueApiException extends IncorrectRequestApiException {
    private NullValueApiException(Translatable detailedMessage) {
        super(detailedMessage);
    }

    static NullValueApiException missedParameter(String parameterName) {
        return new NullValueApiException(ApiFaultTranslations.INSTANCE.detailedMissingParameter(parameterName));
    }

    static NullValueApiException missedField(String fieldPath) {
        return new NullValueApiException(ApiFaultTranslations.INSTANCE.detailedMissingValueOrNull(fieldPath));
    }

    static NullValueApiException missedFieldInArray(String arrayPath, String fieldName) {
        return new NullValueApiException(
                ApiFaultTranslations.INSTANCE.detailedMissingValueOrNullInArray(arrayPath, fieldName));
    }

    /**
     * По переданному {@code violation} определяет, какое детализированное сообщение должно быть.
     * Различаются сценарии:
     * <ul>
     * <li>поле объекта из массива (если родительский элемент &ndash; массив)</li>
     * <li>поле объекта</li>
     * <li>параметр запроса (если родительский элемент отсутствует)</li>
     * </ul>
     * Различие между параметром и полем такое, что поле &ndash; это известный атрибут объекта из запроса.
     * Параметры &ndash; top-level атрибуты и неизвестные атрибуты объектов.
     */
    static NullValueApiException fromConstraintViolation(ConstraintViolation<Object> violation) {
        // Подробнее про то, зачем разбираем путь до поля с ошибкой, в
        // DIRECT-70194: Более понятные сообщения в ошибках парсинга
        Path propertyPath = violation.getPropertyPath();
        String path = simpleViolationPathConverter(propertyPath);
        Path.Node lastNode = Iterables.getLast(propertyPath);
        boolean isInIterable = lastNode.isInIterable();
        if (isInIterable) {
            // Проще по точке разделить название пропущенного поля от пути до него, чем повторять simpleViolationPathConverter
            int pointIdx = path.lastIndexOf('.');
            String field = path.substring(pointIdx + 1);
            String arrayPath = path.substring(0, pointIdx);

            return missedFieldInArray(arrayPath, field);
        } else if (Iterables.size(propertyPath) == 1) {
            // если у проблемного элемента нет родителя, то это параметр, а не поле объекта
            return missedParameter(path);
        } else {
            // не в массиве и не параметр: это просто пропущенное поле
            return missedField(path);
        }
    }
}
