package ru.yandex.direct.api.v5.ws.validation;

import java.util.Arrays;
import java.util.stream.Collectors;

import javax.xml.bind.annotation.XmlEnumValue;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlType;


/**
 * Утилиты для работы с аннотациями, описывающими правила биндинга JAXB
 */
public class XmlBindUtil {
    /**
     * Возвращает описание типа в WSDL документе, соответствующего указанному классу.
     * <p>
     * Для ENUM дополняет название перечислением возможных значений.
     *
     * @param cls класс, WSDL описание которого нужно получить
     * @return строка с описание WSDL типа
     */
    public static String getXmlTypeDescription(Class<?> cls) {
        String xmlTypeName = getXmlTypeName(cls);
        if (cls.isEnum()) {
            return xmlTypeName + " {" + getEnumXmlElements(cls) + "}";
        } else {
            return xmlTypeName;
        }
    }

    /**
     * Возвращает название типа в WSDL документе, соответствующего указанному классу.
     * <p>
     * Название определяется по аннотациям, сгенерированным по WSDL
     *
     * @param cls класс, для которого нужно получить название WSDL типа
     * @return строка с названием типа
     */
    private static String getXmlTypeName(Class<?> cls) {
        XmlType xmlType = cls.getAnnotation(XmlType.class);
        if (xmlType != null && !xmlType.name().isEmpty()) {
            return xmlType.name();
        } else {
            XmlRootElement xmlRootElement = cls.getAnnotation(XmlRootElement.class);
            if (xmlRootElement != null) {
                return xmlRootElement.name();
            } else {
                return cls.getSimpleName();
            }
        }
    }

    /**
     * Возвращает строку с перечислением через запятую элементов Enum в соответствие с WSDL
     *
     * @param enumClass Enum, элементы которого нужно получить
     * @return строка с элементами Enum, разделёнными запятой
     */
    public static String getEnumXmlElements(Class<?> enumClass) {
        return Arrays.stream(enumClass.getEnumConstants()).map(x -> {
            String valueName = ((Enum) x).name();
            try {
                XmlEnumValue xmlEnumValue = enumClass.getField(valueName).getAnnotation(XmlEnumValue.class);
                return xmlEnumValue != null ? xmlEnumValue.value() : valueName;
            } catch (NoSuchFieldException ignored) {
                return valueName;
            }
        }).collect(Collectors.joining(", "));
    }
}
