package ru.yandex.direct.binlogbroker.logbroker_utils.configuration;

import java.time.Duration;
import java.util.Collections;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.LogbrokerCredentialsConfig;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceTypeHelper;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.dbutil.configuration.DbUtilConfiguration;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.TvmClientCredentials;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.misc.thread.ThreadLocalTimeout;

@Configuration
@Import({
        EssentialConfiguration.class,
        DbUtilConfiguration.class
})
@ParametersAreNonnullByDefault
public class LogbrokerCommonConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(LogbrokerCommonConfiguration.class);

    @Bean(destroyMethod = "stop")
    @Lazy
    @Nullable
    public Tvm2 getTvm2(LogbrokerCredentialsConfig logbrokerCredentialsConfig) {
        final int tvmId = logbrokerCredentialsConfig.getTvmClientId();
        final int tvmDstId = logbrokerCredentialsConfig.getTvmDstClientId();
        final String tvmSecret = logbrokerCredentialsConfig.getTvmSecret();
        if (tvmId == -1 || tvmDstId == -1 || tvmSecret == null) {
            logger.warn("Do not use TVM2 auth because TVM2 client_id is {}, dst_client_id is {} and TVM2 secret {}",
                    tvmId,
                    tvmDstId,
                    (tvmSecret == null) ? "was not defined" : "was defined ");  // Do not leak token to logs
            return null;
        } else {
            final Tvm2 tvm2 = new Tvm2(new TvmClientCredentials(tvmId, tvmSecret));
            tvm2.setSrcClientIds(Collections.singletonList(logbrokerCredentialsConfig.getTvmClientId()));
            tvm2.setDstClientIds(Collections.singletonList(logbrokerCredentialsConfig.getTvmDstClientId()));
            // Запускает тред.  Тред останавливается в destroyMethod = "stop" бина.
            tvm2.start();
            return tvm2;
        }

    }

    @Bean
    public LogbrokerCredentialsConfig logbrokerCredentialsConfig(
            @Value("${binlogbroker.logbroker.tvm_secret_path}") String tvmSecretPath,
            @Value("${binlogbroker.logbroker.tvm_client_id}") @Nullable Integer tvmClientId,
            @Value("${binlogbroker.logbroker.tvm_dst_client_id}") @Nullable Integer tvmDstClientId
    ) {
        LogbrokerCredentialsConfig result = new LogbrokerCredentialsConfig();
        if (!StringUtils.isEmpty(tvmSecretPath)) {
            result.setTvmSecretPath(tvmSecretPath);
        }
        if (tvmClientId != null) {
            result.setTvmClientId(tvmClientId);
        }
        if (tvmDstClientId != null) {
            result.setTvmDstClientId(tvmDstClientId);
        }
        return result;
    }

    @Bean
    @Lazy
    public Supplier<Credentials> getLogbrokerCredentialsSupplier(
            final @Nullable Tvm2 tvm2,
            LogbrokerCredentialsConfig logbrokerCredentialsConfig
    ) {
        if (tvm2 == null || logbrokerCredentialsConfig.getTvmDstClientId() == 0) {
            return Credentials.NONE_PROVIDER;
        } else {
            Duration timeout = Duration.ofSeconds(10);  // Взят с потолка
            int tvmDstClientId = logbrokerCredentialsConfig.getTvmDstClientId();
            return () -> ThreadLocalTimeout
                    .withTimeout(
                            org.joda.time.Duration.millis(timeout.toMillis()),
                            () -> tvm2.getServiceTicket(tvmDstClientId))
                    .map(Credentials::tvm)
                    // Старый jdk8 из аркадии не может собрать без указания IllegalStateException в дженерике
                    .getOrThrow(
                            () -> new IllegalStateException("TVM2 service is not initialized for " + timeout));
        }
    }

    @Bean
    @Lazy
    public SourceTypeHelper sourceTypeHelper(ShardHelper shardHelper, EnvironmentType environmentType) {
        return new SourceTypeHelper(shardHelper, environmentType);
    }
}
