package ru.yandex.direct.binlogbroker.logbroker_utils.lock;

import java.time.Duration;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytwrapper.client.YtClusterConfigProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.locks.CypressLockProvider;

import static java.util.Collections.shuffle;

/**
 * Выбирает консьюмера для запуска debug цепочки. Для всех возможных консьюмеров из файла конфигурации пытается взять
 * блокировку в yt. Если хоть одна блокировка взята успешно - debug цепочка будет работать с этим консьюмером
 */
public class ConsumerLockHolder implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ConsumerLockHolder.class);

    private CypressLockProvider.LockImpl consumerLock;
    private final String consumerName;
    private final Consumer<Exception> connectionLostHandler;

    /* Время между успешным пингом и следующим */
    private static final Duration LONG_INTERVAL = Duration.ofSeconds(120);
    /* Время между неуспешным пингом и следующим
     * Не имеет значения, при любом неудачном пинге - завершаем программу
     */
    private static final Duration SHORT_INTERVAL = Duration.ZERO;

    private static final Consumer<Exception> DEFAULT_CONNECTION_LOST_HANDLER = ex -> System.exit(1);


    public ConsumerLockHolder(YtClusterConfigProvider ytClusterConfigProvider, String clusterName,
                              String lockPathPrefixName,
                              List<String> possibleConsumers,
                              Consumer<Exception> connectionLostHandler) {
        var ytCluster = YtCluster.valueOf(clusterName.toUpperCase(Locale.ROOT));
        var clusterConfig = ytClusterConfigProvider.get(ytCluster);
        var lockPath = YPath.simple(clusterConfig.getHome()).child(lockPathPrefixName);
        var cypressLockProvider = new CypressLockProvider(clusterConfig.getProxy(), clusterConfig.getToken(),
                Optional.of(clusterConfig.getUser()), lockPath, LONG_INTERVAL, SHORT_INTERVAL);
        this.consumerName = chooseConsumer(cypressLockProvider, possibleConsumers);
        this.connectionLostHandler = connectionLostHandler;
    }

    public ConsumerLockHolder(YtClusterConfigProvider ytClusterConfigProvider, String clusterName,
                              String lockPathPrefixName,
                              List<String> possibleConsumers) {
        this(ytClusterConfigProvider, clusterName, lockPathPrefixName, possibleConsumers,
                DEFAULT_CONNECTION_LOST_HANDLER);
    }

    public Optional<String> getConsumerName() {
        return Optional.ofNullable(consumerName);
    }

    private String chooseConsumer(CypressLockProvider cypressLockProvider, List<String> possibleConsumers) {
        shuffle(possibleConsumers);

        Exception lastException = null;
        for (var possibleConsumerName : possibleConsumers) {
            try {
                logger.info("Try to get consumerLock for consumer {}", possibleConsumerName);
                consumerLock = cypressLockProvider.createLock(possibleConsumerName, this::handlePingException);
                if (!consumerLock.tryLock()) {
                    logger.info("Failed to take consumerLock for consumer {}: consumer is busy", possibleConsumerName);
                    continue;
                }

                logger.info("Successfully got consumerLock for consumer {}", possibleConsumerName);
                return possibleConsumerName;
            } catch (Exception e) {
                lastException = e;
                logger.warn("Failed to take consumerLock for consumer {}: {}", possibleConsumerName, e);
            }
        }
        if (lastException != null) {
            throw new IllegalStateException("Failed to take consumerLock for any consumer", lastException);
        }

        logger.warn("All consumers are busy: {}", possibleConsumers);
        return null;
    }

    private void handlePingException(Exception e) {
        logger.error("ERROR!!! Consumer lock was lost: {}", e);
        connectionLostHandler.accept(e);
    }


    @Override
    public void close() {
        if (Objects.nonNull(consumerLock) && consumerLock.isTaken()) {
            consumerLock.unlock();
        }
    }
}
