package ru.yandex.direct.binlogbroker.logbroker_utils.models;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;

import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.env.EnvironmentType;

/**
 * Привязка названия источника бинлога, пути к БД внутри db-config и идентификатора группы logbroker.
 */
@ParametersAreNonnullByDefault
public class SourceType {

    /**
     * Полное имя источника, например production:ppc:1
     */
    private final String sourceName;
    /**
     * Имя базы и шард, например ppc:1
     */
    private final String dbName;
    /**
     * Равна шарду для ppc:N, 0 для ppcdict
     */
    private final int logbrokerGroupId;
    private static final int PPCDICT_LOGBROKER_GROUP_ID = 0;

    private SourceType(String sourceName, String dbName, int logbrokerGroupId) {
        this.sourceName = sourceName;
        this.dbName = dbName;
        this.logbrokerGroupId = logbrokerGroupId;
    }

    /**
     * Формирует {@link SourceType} из окружения и имени источника.
     *
     * @param environmentType Окружение, в котором запускается приложение
     * @param sourceName Название источника в формате DB:SHARD или ppcdict.
     * @return Соответствующий названию {@link SourceType}.
     */
    public static SourceType fromType(EnvironmentType environmentType, String sourceName) {
        String[] parts = sourceName.split(":");
        boolean isPpcdict = parts[0].equals("ppcdict");
        Preconditions.checkArgument((parts.length == 1 && isPpcdict) || (parts.length == 2), "Source should be ppcdict or have syntax DB:SHARD");
        int groupId = isPpcdict ? PPCDICT_LOGBROKER_GROUP_ID : Integer.valueOf(parts[1]);
        return new SourceType(environmentType.name().toLowerCase() + ":" + sourceName,
                sourceName, groupId);
    }

    public static SourceType fromPpcShard(EnvironmentType environmentType, int shard) {
        return fromType(environmentType, "ppc:" + shard);
    }

    public int getLogbrokerGroupId() {
        return logbrokerGroupId;
    }

    /**
     * @return Название источника, подобное формату утилиты direct-sql.
     */
    public String getSourceName() {
        return sourceName;
    }

    /**
     * То же самое по смыслу, что и {@link DbConfig#getDbName()}. Для "production:ppc:1" и для "devtest:ppc:1" вернётся
     * "ppc:1".
     */
    public String getDbName() {
        return dbName;
    }

    @Override
    public String toString() {
        return "SourceType{" +
                "sourceName='" + sourceName + '\'' +
                ", dbName='" + dbName + '\'' +
                ", logbrokerGroupId=" + logbrokerGroupId +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof SourceType)) {
            return false;
        }
        SourceType that = (SourceType) o;
        return logbrokerGroupId == that.logbrokerGroupId &&
                Objects.equals(sourceName, that.sourceName) &&
                Objects.equals(dbName, that.dbName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(sourceName, dbName, logbrokerGroupId);
    }
}
