package ru.yandex.direct.binlogbroker.logbroker_utils.models;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.env.EnvironmentType.DEVELOPMENT;
import static ru.yandex.direct.env.EnvironmentType.DEVTEST;

public class SourceTypeHelper {
    private static final Logger logger = LoggerFactory.getLogger(SourceTypeHelper.class);
    private static final String SOURCE_TEMPLATE = "%s:%d";
    private final ShardHelper shardHelper;
    private final EnvironmentType environmentType;

    public SourceTypeHelper(ShardHelper shardHelper, EnvironmentType environmentType) {
        this.shardHelper = shardHelper;
        this.environmentType = environmentType == DEVELOPMENT ? DEVTEST : environmentType;
    }

    /**
     * Возвращает список источников, которые приложение должно читать, в соответствии с настройками
     * и текущим окружением.
     * Deprecated, см. DIRECT-106205
     */
    public Set<SourceType> getSources(@Nullable String sources) {
        throw new IllegalArgumentException("This method is deprecated, see DIRECT-106205 for details");
        // Краткое содержание тикета выше: раньше getSources возвращал только шарды, с добавлением в возможные источники ppcdict'а контракт метода перестал быть понятен. Код, который его использует, не используется.
        // Код метода оставлен для справки или если нужно быстро вернуть.
        //   -- yukaba@
//        if (sources != null && !sources.isEmpty()) {
//            logger.info("Environment: {}, Using sources: {}", environmentType.name(), sources);
//            return StreamEx.of(sources.split(","))
//                    .map(String::trim)
//                    .remove(String::isEmpty)
//                    .map(sourceName -> SourceType.fromType(environmentType, sourceName))
//                    .toImmutableSet();
//        } else {
//            return getAllKnownSourceTypesForEnv();
//        }
    }

    /**
     * Возвращает список источников-шардов ppcdata, которые приложение должно читать, в соответствии с настройками
     * и текущим окружением.
     */
    public Set<SourceType> getPpcSources(@Nonnull List<Integer> shards) {
        if (!shards.isEmpty()) {
            logger.info("Environment: {}, Shards: {}", environmentType.name(), shards);
            return shards.stream()
                    .map(shard -> String.format(SOURCE_TEMPLATE, "ppc", shard))
                    .map(sourceName -> SourceType.fromType(environmentType, sourceName))
                    .collect(toSet());
        } else {
            Set<SourceType> allKnownSourceTypes = new HashSet<>(getAllKnownSourceTypesForEnv());
            allKnownSourceTypes.remove("ppcdict");
            return allKnownSourceTypes;
        }
    }

    /**
     * Возвращает источник, соответствующий базе ppcdict в текущем окружении.
     */
    public SourceType getPpcdictSource() {
        return SourceType.fromType(environmentType, "ppcdict");
    }

    /**
     * Возвращает все известные источники бинлога, которые соответствуют желаемой среде.
     */
    private Set<SourceType> getAllKnownSourceTypesForEnv() {
        List<String> sourceNames = StreamEx.of(shardHelper.dbShards())
                .map(shard -> String.format(SOURCE_TEMPLATE, "ppc", shard))
                .toList();
        sourceNames.add("ppcdict");
        logger.info("Environment: {}, Using all known sources: {}",
                environmentType.name(), Arrays.toString(sourceNames.toArray()));
        return StreamEx.of(sourceNames)
                .map(sourceName -> SourceType.fromType(environmentType, sourceName))
                .toImmutableSet();
    }
}
