package ru.yandex.direct.binlogbroker.logbroker_utils.reader;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.Interrupts;

/**
 * Сейчас использовать этот класс можно, только если коммит присходит после кажой попытки чтения
 * https://st.yandex-team.ru/DIRECT-100449 (п2)
 */
@ParametersAreNonnullByDefault
public class RetryingLogbrokerBatchReader<T> implements LogbrokerBatchReader<T> {
    private static final Logger logger = LoggerFactory.getLogger(RetryingLogbrokerBatchReader.class);
    private static final long DEFAULT_SLEEP_TIME = 5000;  // 5000ms = 5sec
    private static final int DEFAULT_RETRY_COUNT = 10;

    private final Supplier<LogbrokerBatchReader<T>> eventReaderSupplier;
    private final int retryCount;
    @Nullable
    private LogbrokerBatchReader<T> eventReader;
    private long sleepTime;
    private volatile boolean closed = false;

    public RetryingLogbrokerBatchReader(Supplier<LogbrokerBatchReader<T>> eventReaderSupplier) {
        this(eventReaderSupplier, DEFAULT_RETRY_COUNT);
    }

    public RetryingLogbrokerBatchReader(Supplier<LogbrokerBatchReader<T>> eventReaderSupplier, int retryCount) {
        this(eventReaderSupplier, retryCount, DEFAULT_SLEEP_TIME);
    }

    public RetryingLogbrokerBatchReader(Supplier<LogbrokerBatchReader<T>> eventReaderSupplier, int retryCount,
                                        long sleepTime) {
        this.sleepTime = sleepTime;
        this.eventReaderSupplier = eventReaderSupplier;
        this.retryCount = retryCount;
        this.eventReader = createEventReader(eventReaderSupplier, retryCount, sleepTime).left;
    }

    private ImmutablePair<LogbrokerBatchReader<T>, Integer> createEventReader(
            Supplier<LogbrokerBatchReader<T>> eventReaderSupplier, int retries, long sleepTime) {
        while (true) {
            try {
                LogbrokerBatchReader<T> reader = eventReaderSupplier.get();
                return ImmutablePair.of(reader, retries);
            } catch (LogbrokerReaderInitException ex) {
                --retries;
                if (retries > 0) {
                    logger.error("Failed to create event reader, {} attempts left", retries, ex);
                } else {
                    throw ex;
                }
            }
            sleep(sleepTime);
        }
    }

    private static void sleep(long millis) {
        Interrupts.failingRun(
                () -> Thread.sleep(millis)
        );
    }

    @Override
    public void fetchEvents(Interrupts.InterruptibleFunction<List<T>, LogbrokerCommitState> eventsConsumer)
            throws InterruptedException {
        int retires = this.retryCount;

        while (!closed) {
            if (eventReader == null) {
                ImmutablePair<LogbrokerBatchReader<T>, Integer> pair =
                        createEventReader(eventReaderSupplier, retires, sleepTime);
                retires = pair.right;
                eventReader = pair.left;
            }
            try {
                eventReader.fetchEvents(eventsConsumer);
                return;
            } catch (LogbrokerReaderException ex) {
                LogbrokerBatchReader<T> failedEventReader = eventReader;
                eventReader = null;
                --retires;
                if (retires > 0) {
                    logger.warn("Failed to get data from logbroker, {} attempts left", retires, ex);
                }
                try {
                    failedEventReader.close();
                } catch (Exception ex1) {
                    logger.warn("Failed to close logbroker reader. Ignoring.", ex1);
                }
                if (retires <= 0) {
                    throw ex;
                }
            }
            sleep(sleepTime);
        }
    }

    @Override
    public void close() throws LogbrokerReaderCloseException {
        if (eventReader != null && !closed) {
            closed = true;
            eventReader.close();
        }
    }
}
