package ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiPredicate;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderException;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState.DONT_COMMIT;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

@ParametersAreNonnullByDefault
public abstract class LogbrokerBatchReaderImpl<T> implements LogbrokerBatchReader<T> {
    private static final Logger logger = LoggerFactory.getLogger(LogbrokerBatchReaderImpl.class);

    private final SyncConsumer logbrokerConsumer;
    private final boolean logbrokerNoCommit;
    private final LogbrokerReadingStrategy<T> logbrokerReadingStrategy;

    private final Counter readingObjects;
    private final GaugeInt64 readingDuration;
    private final Counter readingBytes;
    private final GaugeInt64 commitDuration;

    protected Duration iterationTime = Duration.ofSeconds(5);

    /**
     * Для корректной работы LogBroker при commit необходимо прислать все cookie отданные во время чтений.
     * Поэтому, в этом списке накапливаются cookies до принятие решения о том, что можно делать commit.
     */
    private final List<Long> logbrokerCookies = new ArrayList<>();

    public LogbrokerBatchReaderImpl(Supplier<SyncConsumer> logbrokerConsumerSupplier, boolean logbrokerNoCommit) {
        this(logbrokerConsumerSupplier, logbrokerNoCommit, null);
    }

    public LogbrokerBatchReaderImpl(Supplier<SyncConsumer> logbrokerConsumerSupplier, boolean logbrokerNoCommit,
                                    boolean needReadingOptimization) {
        this(logbrokerConsumerSupplier, logbrokerNoCommit, null, needReadingOptimization);
    }

    public LogbrokerBatchReaderImpl(Supplier<SyncConsumer> logbrokerConsumerSupplier, boolean logbrokerNoCommit,
                                    @Nullable MetricRegistry metricRegistry) {
        this(logbrokerConsumerSupplier, logbrokerNoCommit, metricRegistry, false);
    }

    /**
     * @param metricRegistryNullable в какой registry нужно отправлять метрики, если null, то будут отправляться в
     *                                основной - {@link SolomonUtils#SOLOMON_REGISTRY}
     * @param needReadingOptimization нужна ли оптимизация чтения. Если включена, то будет выбрана стратегрия чтения
     *                                {@link OptimalLogbrokerReadingStrategy}. Рекомендуется, если чтение-обработка
     *                                происходят
     *                                в цикле
     */
    public LogbrokerBatchReaderImpl(Supplier<SyncConsumer> logbrokerConsumerSupplier, boolean logbrokerNoCommit,
                                    @Nullable MetricRegistry metricRegistryNullable,
                                    boolean needReadingOptimization) {
        this.logbrokerConsumer = logbrokerConsumerSupplier.get();
        this.logbrokerNoCommit = logbrokerNoCommit;
        this.logbrokerReadingStrategy = needReadingOptimization ?
                new OptimalLogbrokerReadingStrategy<>(this::readAndParse) :
                new SimpleLogbrokerReadingStrategy<>(this::readAndParse);
        var metricRegistry = Objects.isNull(metricRegistryNullable) ? SOLOMON_REGISTRY : metricRegistryNullable;
        this.readingObjects = metricRegistry.counter("reading_objects_count");
        this.readingDuration = metricRegistry.gaugeInt64("reading_duration");
        this.readingBytes = metricRegistry.counter("reading_bytes");
        this.commitDuration = metricRegistry.gaugeInt64("commit_duration");

    }

    /**
     * Fetch data from logbroker, and feed to consumer.
     * If consumer throws no exception, and returned value is not DONT_COMMIT
     * do a logbroker commit (i.e. mark data as consumed).
     * <p>
     * Groups several logbroker batches into one big batch, applies big batch and commits all cookies of applied
     * batches.
     * <p>
     * To improve speed, event handling and commit are work asynchronously.
     * Their wait occurs after the next reading and parsing of events.
     *
     * @param eventsConsumer consumer to feed data to
     */
    @Override
    public void fetchEvents(Interrupts.InterruptibleFunction<List<T>, LogbrokerCommitState> eventsConsumer) throws InterruptedException {
        var eventsWithCookies = logbrokerReadingStrategy.readAndParse();
        logbrokerCookies.addAll(eventsWithCookies.cookies);
        if (eventsWithCookies.events.isEmpty()) {
            return;
        }
        handleAndCommit(eventsWithCookies.events, eventsConsumer);
    }

    protected abstract List<T> batchDeserialize(MessageBatch messageBatch);

    protected abstract int count(List<T> e);

    /**
     * Решает, можно ли продолжить буфферизацию событий или следует записать то, что в буффере.
     * Параметры - общее количество строк, уже собранное в буффер, время, затраченное на сбор этих строк
     * и количество байт в прочитанных сообщениях.
     * Если метод не переопределен - вызывается {@link #batchingThreshold}
     * Должен вернуть true, если можно продолжить буфферизацию и false, если следует немедленно обработать буффер.
     */
    protected BatchingThresholdPredicate<Long, Duration, Long> batchingThresholdPredicate() {
        return (rows, time, bytes) -> batchingThreshold().test(rows, time);
    }

    /**
     * Аналогично {@link #batchingThresholdPredicate()}, но принимает только количество строк и время,
     * затраченное на сбор этих строк.
     */
    protected BiPredicate<Long, Duration> batchingThreshold() {
        return (rows, time) -> rows < 10_000 && time.compareTo(iterationTime) < 0;
    }

    private EventsWithCookies<T> readAndParse() {
        List<T> events = new ArrayList<>();
        List<Long> localCookies = new ArrayList<>();
        AtomicLong currentReadingObjects = new AtomicLong();
        long currentReadingBytes = 0;
        MonotonicTime startTime = NanoTimeClock.now();
        BatchingThresholdPredicate<Long, Duration, Long> batchingThreshold = batchingThresholdPredicate();

        try (TraceProfile profile = Trace.current().profile("logbroker_batch_reader.fetch_events")) {
            if (logger.isDebugEnabled()) {
                logger.debug("Trying to read data from {}", this.logbrokerConsumer);
            }

            ConsumerReadResponse response;
            do {
                response = Interrupts.failingGet(this.logbrokerConsumer::read);
                if (response == null) {
                    break;  // No data received, because of timeout or other reason.
                }
                localCookies.add(response.getCookie());

                List<MessageBatch> batches = response.getBatches();

                currentReadingBytes += batches.stream()
                        .flatMap(messageBatch -> messageBatch.getMessageData().stream())
                        .mapToLong(messageData -> messageData.getRawData().length)
                        .sum();

                logger.debug("Logbroker reader batch size {}", batches.size());

                batches.stream()
                        .map(this::batchDeserialize)
                        .filter(Objects::nonNull)
                        .forEachOrdered(e -> {
                            events.addAll(e);
                            currentReadingObjects.addAndGet(count(e));
                        });
            } while (batchingThreshold.test(currentReadingObjects.get(), NanoTimeClock.now().minus(startTime),
                    currentReadingBytes));
        }
        long currentReadingDuration = NanoTimeClock.now().minus(startTime).getSeconds();
        if (logger.isDebugEnabled()) {
            logger.debug("Got {} events ({} rows total), {} bytes from logbroker for {} sec",
                    events.size(), currentReadingObjects.get(), currentReadingBytes, currentReadingDuration);
        }
        if (!events.isEmpty()) {
            readingObjects.add(currentReadingObjects.get());
            readingDuration.set(currentReadingDuration);
            readingBytes.add(currentReadingBytes);
        }
        return new EventsWithCookies<>(events, localCookies);
    }

    private void handleAndCommit(List<T> events,
                                 Interrupts.InterruptibleFunction<List<T>, LogbrokerCommitState> eventsConsumer) throws InterruptedException {
        try (TraceProfile profile = Trace.current().profile("logbroker_batch_reader.apply_and_commit")) {
            LogbrokerCommitState logbrokerCommitState = eventsConsumer.apply(events);
            if (Objects.equals(logbrokerCommitState, DONT_COMMIT) || logbrokerNoCommit) {
                return;
            }

            long startCommit = getTimestamp();
            try {
                logbrokerConsumer.commit(logbrokerCookies);
            } catch (TimeoutException | RuntimeException ex) {
                throw new LogbrokerReaderException(ex);
            }
            commitDuration.set(getTimestamp() - startCommit);
            logbrokerCookies.clear();
        }
    }


    /**
     * @throws LogbrokerReaderCloseException, если во время закрытия logbrokerConsumer произошло исключение
     */
    @Override
    public void close() throws LogbrokerReaderCloseException {
        try {
            logbrokerConsumer.close();
        } catch (Exception e) {
            logger.warn("Error while closing logbroker sync consumer: {}", e);
            throw new LogbrokerReaderCloseException(e);
        } finally {
            logbrokerReadingStrategy.close();
        }
    }

    private long getTimestamp() {
        return System.currentTimeMillis() / 1000;
    }

    static class EventsWithCookies<T> {
        List<T> events;
        List<Long> cookies;

        EventsWithCookies(List<T> events, List<Long> cookies) {
            this.events = events;
            this.cookies = cookies;
        }
    }
}
