package ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.function.Supplier;

import com.google.common.util.concurrent.ThreadFactoryBuilder;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.misc.thread.ExecutionRuntimeException;

/**
 * Выполняет читение блока данных в отдельном потоке.
 * При запросе результата чтения дожидается окончания чтения, возвращает результат и тут же запускает чтения нового
 * блока данных
 */
public class OptimalLogbrokerReadingStrategy<T> implements LogbrokerReadingStrategy<T> {
    private final ExecutorService readAndParseExecutionService;
    private CompletableFuture<LogbrokerBatchReaderImpl.EventsWithCookies<T>> readAndParseFuture;
    private final Supplier<LogbrokerBatchReaderImpl.EventsWithCookies<T>> readAndParseCallback;

    OptimalLogbrokerReadingStrategy(Supplier<LogbrokerBatchReaderImpl.EventsWithCookies<T>> readAndParseCallback) {
        this.readAndParseCallback = readAndParseCallback;
        ThreadFactory threadFactory =
                new ThreadFactoryBuilder().setNameFormat("read-and-parse-executor-%02d").setDaemon(true).build();
        this.readAndParseExecutionService = Executors.newSingleThreadExecutor(threadFactory);
    }

    @Override
    public LogbrokerBatchReaderImpl.EventsWithCookies<T> readAndParse() {
        LogbrokerBatchReaderImpl.EventsWithCookies<T> currentResult;
        if (Objects.isNull(readAndParseFuture)) {
            restartReadAndParseFuture();
        }
        try {
            currentResult = readAndParseFuture.get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        } catch (ExecutionException e) {
            throw new ExecutionRuntimeException(e);
        }
        restartReadAndParseFuture();
        return currentResult;
    }

    private void restartReadAndParseFuture() {
        Trace currentTrace = Trace.current();
        Supplier<LogbrokerBatchReaderImpl.EventsWithCookies<T>> readAndParseCallbackWithTrace = () -> {
            Trace.push(currentTrace);
            LogbrokerBatchReaderImpl.EventsWithCookies<T> result;
            try {
                result = readAndParseCallback.get();
            } finally {
                Trace.pop();
            }
            return result;
        };
        readAndParseFuture = CompletableFuture.supplyAsync(readAndParseCallbackWithTrace, readAndParseExecutionService);
    }

    @Override
    public void close() {
        readAndParseExecutionService.shutdown();
    }
}
