package ru.yandex.direct.binlogbroker.logbroker_utils.writer;

import java.io.Closeable;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerInitResponse;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerWriteResponse;

/**
 * Асинхронно-синхронный отправитель сообщений в логброкер. Если объем сообщений, ранее отправленных в логброкер,
 * но по которым еще не пришло подтверждение (сообщения "в полете") меньше чем указанный размер буфера, то отправляет
 * сообщения асинхронно, если же объем сообщений "в полете" больше чем размер буфера, то синхронно ждет подтверждения
 * отправки от логброкера уже имеющихся сообщений, пока объем сообщений "в полете" не станет меньше размера буфера,
 * и только потом отправляет новое сообщение в логброкер.
 * @see <a href="https://logbroker.yandex-team.ru/docs/concepts/data/write#best-practices">
 *     Документация Logroker: Рекомендации по эффективной записи</a>
 * @param <TMessage> Тип сообщения для отправки
 * @param <TMessageId> Тип идентификатора сообщений для отправки. В процессе работы можно будет получить идентификатор
 *                    последнего подтвержденного записанного сообщения в логброкер
 */
@ParametersAreNonnullByDefault
public abstract class AbstractBufferedLogbrokerWriter<TMessage, TMessageId> implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(AbstractBufferedLogbrokerWriter.class);
    private static final byte[] ZERO_BYTES = new byte[0];

    private long currentBytesOnTheFly;
    private final long maxBytesOnTheFly;
    private final Queue<MessageTaskInfo<TMessageId>> writeQueue;
    private final Object locker = new Object();
    private boolean stopped;
    private CompletableFuture<ProducerWriteResponse> waitingFuture = null;
    private final Duration logbrokerTimeout;
    private final Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier;
    private AsyncProducer currentLogbrokerProducer;
    private long maxSecNo;

    private TMessageId lastWrittenMessageId;

    // нужны только для статистики
    private long totalBytesCompleted;
    private long totalBytes;
    private int totalMessagesCompleted;
    private int emptyMessagesCompleted;
    private int totalMessages;
    private int emptyMessages;

    public AbstractBufferedLogbrokerWriter(
            Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
            Duration logbrokerTimeout,
            long maxBytesOnTheFly) {
        this.maxBytesOnTheFly = maxBytesOnTheFly;
        this.logbrokerTimeout = logbrokerTimeout;
        this.logbrokerProducerSupplier = logbrokerProducerSupplier;
        this.writeQueue = new ArrayDeque<>();
    }

    /**
     * Конвертирует сообщение в набор байт. Не вызывается для пустых сообщений
     * @param message сообщение для конвертации. Не может быть null
     * @return сконвертированное сообщение. Не может быть null.
     */
    protected abstract byte[] convertToBytes(TMessage message);

    /**
     * Может обрабатывать ответ от логброкера, например кидать эксепшн, если ответ неожиданный. Для изначально
     * пустых сообщений, переданный в метод {@link #write(Object, Object)}, будет вызван со значением
     * producerWriteResponse равным <code>new ProducerWriteResponse(-1, -1, false)</code>.
     * Реализация по-умолчанию ничего не делает.
     * @param producerWriteResponse ответ о записи сообщения от логброкера или
     *                              <code>new ProducerWriteResponse(-1, -1, false)</code> для пустых сообщений.
     * @param messageId идентификатор подтвержденного сообщения
     */
    protected void acceptLogbrokerResponse(ProducerWriteResponse producerWriteResponse, TMessageId messageId) { }

    /**
     * Получает номер следующего сообщения в логброкер. Не вызывается для пустых сообщений
     * @param message сообщение для отправки в логброкер, не может быть null
     * @param messageId идентификатор отправляемого сообщения
     * @return реализация по-умолчанию возвращает следующий порядковый номер сообщения (номер предыдущего + 1)
     */
    protected long getNextSeqNo(TMessage message, TMessageId messageId) {
        return ++maxSecNo;
    }

    /**
     * Инициализирует поставщика данный в логброкер, если он еще не инициализирован. Метод имеет смысл
     * вызывать отдельно, перед началом записи, только если требуется получить идентификационный номер
     * последнего сохраненного в данном sourceId сообщения. Если этот номер не нужен, то можно метод не вызывать,
     * инициализация произойдет при первом обращении к методу {@link #write(Object, Object)}.
     * @return идентификационный номер последнего сохраненного в данном sourceId сообщения.
     * @throws ExecutionException при ошибке внутри задач по инициализации поставщика данных в логброкер
     * @throws TimeoutException при таймауте ожидания результата задачи по инициализации поставщика данных в логброкер
     * @throws InterruptedException при прерывании текущего потока, во время ожидания результата задачи
     * по инициализации поставщика данных в логброкер
     */
    public long init() throws ExecutionException, TimeoutException, InterruptedException {
        synchronized (locker) {
            getCurrentLogbrokerProducer();
            return maxSecNo;
        }
    }

    /**
     * Лениво инициализирует поставщика данный в логброкер, если он еще не инициализирован. Должно вызываться
     * в критической секции на {@link #locker}
     * @return текущий поставщик данный в логброкер
     * @throws ExecutionException при ошибке внутри задач по инициализации поставщика данных в логброкер
     * @throws TimeoutException при таймауте ожидания результата задачи по инициализации поставщика данных в логброкер
     * @throws InterruptedException при прерывании текущего потока, во время ожидания результата задачи
     * по инициализации поставщика данных в логброкер
     */
    private AsyncProducer getCurrentLogbrokerProducer()
            throws ExecutionException, TimeoutException, InterruptedException {
        if (currentLogbrokerProducer == null) {
            currentLogbrokerProducer =
                    logbrokerProducerSupplier.get().get(logbrokerTimeout.toMillis(), TimeUnit.MILLISECONDS);
            ProducerInitResponse initResponse = currentLogbrokerProducer.init()
                    .get(logbrokerTimeout.toMillis(), TimeUnit.MILLISECONDS);
            maxSecNo = initResponse.getMaxSeqNo();
        }
        return currentLogbrokerProducer;
    }

    /**
     * Отправляет сообщения в логброкер. Если объем сообщений, ранее отправленных в логброкер, но по которым еще
     * не пришло подтверждение (сообщения "в полете") меньше чем размер буфера, делает это асинхронно, если же
     * объем сообщений "в полете" больше чем размер буфера, то синхронно ждет подтверждения отправки от логброкера
     * уже имеющихся сообщений, пока объем сообщений "в полете" не станет меньше размера буфера, и только потом
     * отправляет указанное сообщение в логброкер.
     *
     * Иногда сообщения приходят пустые. Это делается в том случае, если отправлять сообщение в логброкер не надо,
     * а сохранять его идентификатор, как идентификатор последнего обработанного сообщения где-то надо.
     * Поэтому, если приходит пустое сообщение, то метод положит его в очередь уже "выполненным", с нулевым размером.
     * Это позволит, когда до такого сообщения дойдет очередь, корректно установить {@link #lastWrittenMessageId},
     * и не потерять обработанный messageId пустого сообщения.
     * @param message Сообщение для отправки в логброкер. Может быть пустым
     * @param messageId Идентификатор сообщения. Когда сообщение будет подтверждено логброкером как принятое,
     *                  это значение будет установлено как {@link #lastWrittenMessageId}
     * @throws ExecutionException при ошибке внутри задачи по отправке сообщения в логброкер
     * @throws TimeoutException при таймауте ожидания результата задачи по отправке сообщения в логброкер
     * @throws InterruptedException при прерывании текущего потока, во время ожидания результата задачи по отправке
     * сообщения в логброкер
     */
    public void write(@Nullable TMessage message, TMessageId messageId) throws
            ExecutionException, TimeoutException, InterruptedException {
        // Сериализуем сообщение, если оно не пустое
        byte[] bytes = message == null ? ZERO_BYTES : convertToBytes(message);
        // Ждем, пока размер буфера не станет достаточным для записи переданного сообщения
        // (или не ждем, если он уже достаточен)
        trackTasksToBufferSize(Math.max(0, maxBytesOnTheFly - bytes.length));
        synchronized (locker) {
            // Остановимся, если попросили
            if (stopped || Thread.currentThread().isInterrupted()) {
                return;
            }
            CompletableFuture<ProducerWriteResponse> writeTask;
            long secNo;
            // Создаем новую задачу по отправке сообщения в логброкер
            if (message == null) {
                // Сообщение пустое. Создаем задание для него как уже выполненое
                secNo = -1;
                writeTask = CompletableFuture.completedFuture(
                        new ProducerWriteResponse(secNo, -1, false));
                emptyMessages++;
            } else {
                // Сообщение не пустое, честно отправляем его в логброкер
                AsyncProducer writer = getCurrentLogbrokerProducer();
                secNo = getNextSeqNo(message, messageId);
                writeTask = writer.write(bytes, secNo);
                currentBytesOnTheFly += bytes.length;
                totalBytes += bytes.length;
            }
            totalMessages++;
            MessageTaskInfo<TMessageId> batchInfo = new MessageTaskInfo<>(writeTask, secNo, messageId, bytes.length);
            writeQueue.add(batchInfo);
        }
    }

    /**
     * Прокручивает задачи в очереди до тех пор, пока задачи не кончатся,
     * либо размер буфера станет меньше maxBufferSizeInBytes
     * @param maxBufferSizeInBytes максимальный размер буфера ожидающих задач в байтах
     * @throws ExecutionException при ошибке внутри задачи по отправке сообщения в логброкер
     * @throws TimeoutException при таймауте ожидания результата задачи по отправке сообщения в логброкер
     * @throws InterruptedException при прерывании текущего потока, во время ожидания результата задачи по отправке
     * сообщения в логброкер
     */
    private void trackTasksToBufferSize(long maxBufferSizeInBytes)
            throws ExecutionException, TimeoutException, InterruptedException {
        while (true) {
            MessageTaskInfo<TMessageId> messageTaskInfo;
            synchronized (locker) {
                messageTaskInfo = writeQueue.peek();
                // Если в голове очереди есть задача, то обрабатываем ее, только если у нас закончился буфер,
                // или задача полностью выполнена.
                boolean canTrack = messageTaskInfo != null &&
                        (currentBytesOnTheFly > maxBufferSizeInBytes || messageTaskInfo.writeTask.isDone());
                if (!canTrack) {
                    // Если обрабатывать задачу не нужно (ее нет, или буфера хватает, а задача еще не завершена),
                    // то выходим из ожидания.
                    break;
                }
                // Если надо обработать - вынимаем задачу из очереди
                messageTaskInfo = writeQueue.poll();
                if (messageTaskInfo == null) {
                    throw new IllegalStateException(
                            "Not null message task info was peeked from queue, but null polled in critical section");
                }
                // Текущая задача, которую мы ждем. Она кэнселится в close(), чтобы снять блок ожидания
                waitingFuture = messageTaskInfo.writeTask;
            }
            if (logger.isTraceEnabled() && !waitingFuture.isDone()) {
                logger.trace(String.format(
                        "Possible waiting on task with secNo: %d, messageId: %s, bytes count: %d, timeout: %.3f sec, " +
                                "current buffer: %.3f MB, max current allowed buffer: %.3f MB",
                        messageTaskInfo.secNo, messageTaskInfo.messageId, messageTaskInfo.bytesCount,
                        logbrokerTimeout.toMillis() / 1000.0,
                        currentBytesOnTheFly / (1024.0 * 1024.0), maxBufferSizeInBytes / (1024.0 * 1024.0)));
            }
            // Ожидаем результат задачи. Если она уже завершена, то проскочим сразу, а если нет,
            // то значит у нас закончился буфер и надо подождать завершения этой задачи, перед тем,
            // как добавлять новые задачи в очередь. Ожидание может быт прервано методом close();
            ProducerWriteResponse logbrokerWriteResponse =
                    waitingFuture.get(logbrokerTimeout.toMillis(), TimeUnit.MILLISECONDS);
            // Изменяем идентификатор последнего подтвержденного сообщения
            lastWrittenMessageId = messageTaskInfo.messageId;
            // Отправляем результат подтверждения на проверку
            acceptLogbrokerResponse(logbrokerWriteResponse, lastWrittenMessageId);
            // Изменяем размер буфера
            currentBytesOnTheFly -= messageTaskInfo.bytesCount;
            // Обновляем статистику
            totalBytesCompleted += messageTaskInfo.bytesCount;
            totalMessagesCompleted++;
            if (messageTaskInfo.bytesCount == 0) {
                emptyMessagesCompleted++;
            }
        }
    }

    /**
     * Синхронно ожидает до тех пор, пока от логброкера не придет подтверждение по всем отправленным в него сообщениям
     * @throws ExecutionException при ошибке внутри задачи по отправке сообщения в логброкер
     * @throws TimeoutException при таймауте ожидания результата задачи по отправке сообщения в логброкер
     * @throws InterruptedException при прерывании текущего потока, во время ожидания результата задачи по отправке
     * сообщения в логброкер
     */
    public void waitUntilAllWritten() throws ExecutionException, TimeoutException, InterruptedException {
        trackTasksToBufferSize(-1);
    }

    /**
     * Возвращает идентификатор последнего подтвержденного записанного сообщения в логброкер
     * (или последнего обработанного пустого сообщения). Идентификаторы, возвращаемые этим методом, всегда будут
     * идти в том же порядке, в котором попадали в метод {@link #write(Object, Object)}
     * @return идентификатор последнего подтвержденного записанного сообщения в логброкер (или последнего обработанного
     * пустого сообщения). Может быть равен null, если еще ничего не обработано.
     */
    public TMessageId getLastWrittenMessageId() {
        return lastWrittenMessageId;
    }

    /**
     * Возвращает строку со статистикой об объеме буфера, отравленных сообщений, и их количеству
     * @return строка со статистикой об объеме буфера, отравленных сообщений, и их количеству
     */
    public String getStateAsString() {
        return String.format(
                "Bytes. buffer: %.3f MB, completed: %.3f MB, total: %.3f MB | " +
                        "Messages/Non Empty. completed: %d, total: %d | " +
                        "Messages/Total. queue size: %d, completed: %d, total: %d, empty: %d",
                currentBytesOnTheFly / (1024.0 * 1024),
                totalBytesCompleted / (1024.0 * 1024),
                totalBytes / (1024.0 * 1024),
                totalMessagesCompleted - emptyMessagesCompleted, totalMessages - emptyMessages,
                writeQueue.size(), totalMessagesCompleted, totalMessages, emptyMessages);
    }

    /**
     * Закрывает соединение с логброкером и отменяет все имеющиеся невыполненные задачи по отправке сообщений в
     * логброкер.
     */
    @Override
    @PreDestroy
    public void close() {
        synchronized (locker) {
            if (stopped) {
                return;
            }
            stopped = true;
            if (currentLogbrokerProducer != null) {
                currentLogbrokerProducer.close();
            }
            if (waitingFuture != null && !waitingFuture.isDone()) {
                logger.warn("Current waiting future is not done");
                waitingFuture.cancel(true);
            }
            if (writeQueue.size() > 0) {
                logger.warn("Writing queue size: {}", writeQueue.size());
                while (!writeQueue.isEmpty()) {
                    MessageTaskInfo<TMessageId> messageTaskInfo = writeQueue.poll();
                    if (messageTaskInfo == null) {
                        logger.warn("Polled null message task info from non empty queue");
                    } else {
                        messageTaskInfo.writeTask.cancel(true);
                    }
                }
            }
        }
    }

    /**
     * Внутренний класс для хранения информации об отправляемом сообщении
     * @param <TMessageId> тип идентификатора отправляемого сообщения
     */
    private static class MessageTaskInfo<TMessageId> {
        private final CompletableFuture<ProducerWriteResponse> writeTask;
        private final long secNo;
        private final TMessageId messageId;
        private final int bytesCount;

        public MessageTaskInfo(
                CompletableFuture<ProducerWriteResponse> writeTask, long secNo, TMessageId messageId, int bytesCount) {
            this.writeTask = writeTask;
            this.secNo = secNo;
            this.messageId = messageId;
            this.bytesCount = bytesCount;
        }
    }
}
