package ru.yandex.direct.binlogbroker.logbroker_utils.writer;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerInitResponse;

import static java.util.concurrent.TimeUnit.SECONDS;
import static ru.yandex.direct.binlogbroker.logbroker_utils.writer.CompletableFutureUtils.wrapCompletableFutureToRetry;
import static ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriterRetryConfig.defaultConfig;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public abstract class AbstractLogbrokerWriterImpl<D> implements LogbrokerWriter<D> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractLogbrokerWriterImpl.class);

    private final Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier;
    private final long logbrokerTimeoutSec;
    private final int retryCount;
    private final Duration retryDelay;

    private volatile CompletableFuture<AsyncProducer> logbrokerProducerFuture;
    private volatile Long initSeqNo;
    private AtomicLong currentSeqNo = new AtomicLong();

    private CompletableFuture<Integer> currentWriteFuture = CompletableFuture.completedFuture(null);
    private volatile boolean isClosed = false;
    private final ReentrantLock closeLock = new ReentrantLock();
    private static final boolean DEFAULT_LAZY_INIT = true;

    public AbstractLogbrokerWriterImpl(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                       Duration logbrokerTimeout,
                                       LogbrokerWriterRetryConfig logbrokerWriterRetryConfig, boolean lazyInit) {
        this.logbrokerProducerSupplier = logbrokerProducerSupplier;
        this.logbrokerTimeoutSec = logbrokerTimeout.getSeconds();
        this.retryCount = logbrokerWriterRetryConfig.getRetryCount();
        this.retryDelay = logbrokerWriterRetryConfig.getRetryDelay();
        if (!lazyInit) {
            initSync();
        }
    }

    public AbstractLogbrokerWriterImpl(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                       Duration logbrokerTimeout,
                                       LogbrokerWriterRetryConfig logbrokerWriterRetryConfig) {
        this(logbrokerProducerSupplier, logbrokerTimeout, logbrokerWriterRetryConfig, DEFAULT_LAZY_INIT);
    }

    public AbstractLogbrokerWriterImpl(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                       Duration logbrokerTimeout, boolean lazyInit) {
        this(logbrokerProducerSupplier, logbrokerTimeout, defaultConfig(), lazyInit);
    }

    public AbstractLogbrokerWriterImpl(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                       Duration logbrokerTimeout) {
        this(logbrokerProducerSupplier, logbrokerTimeout, DEFAULT_LAZY_INIT);
    }

    protected abstract LogbrokerWriteRequest makeRequest(D record);


    @Override
    public int writeSync(List<D> records) {
        CompletableFuture<Integer> completableFuture = write(records);
        try {
            return completableFuture.get(logbrokerTimeoutSec, SECONDS);
        } catch (ExecutionException | TimeoutException ex) {
            throw new LogBrokerWriterException(ex);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    /**
     * Если запись просиходит с попыткой перезаписи в случае неудачи, то перед выполнением записи ожидается
     * завершение предыдущей
     * так как при повторе записи данные могут записаться в неверном порядке и проихойдет потеря данных
     * Если запись происходит без попытки перезаписи(в конструкторе передан retryCnt = 0), то такое поведение не
     * требуется
     */
    @Override
    public CompletableFuture<Integer> write(List<D> records) {
        Supplier<CompletableFuture<Integer>> writeSupplier = () -> createAndInitProducer()
                .thenCompose(
                        unused -> wrapCompletableFutureToRetry(() -> writeImpl(records),
                                this::restartLogbrokerProducer,
                                retryCount, retryDelay));
        if (retryCount != 0) {
            currentWriteFuture = currentWriteFuture.thenCompose(unused -> writeSupplier.get());
            return currentWriteFuture;
        } else {
            return writeSupplier.get();
        }
    }

    private CompletableFuture<AsyncProducer> createAndInitProducer() {
        if (logbrokerProducerFuture != null) {
            return logbrokerProducerFuture;
        }
        logger.debug("Close lock are going to be received by createAndInitProducer method");
        closeLock.lock();
        try {
            if (isClosed) {
                throw new LogBrokerWriterInitException("Logbroker writer closed");
            }
            CompletableFuture<AsyncProducer> createProducerFuture =
                    wrapCompletableFutureToRetry(logbrokerProducerSupplier, retryCount, retryDelay);

            logbrokerProducerFuture = createProducerFuture.thenCompose(
                    asyncProducer -> wrapCompletableFutureToRetry(asyncProducer::init, retryCount, retryDelay))
                    .thenAccept(this::acceptInitResponse)
                    .thenCompose(unused -> createProducerFuture);
        } finally {
            closeLock.unlock();
            logger.debug("Close lock released by createAndInitProducer method");
        }
        return logbrokerProducerFuture;
    }

    private void initSync() {
        try {
            createAndInitProducer().get(logbrokerTimeoutSec, SECONDS);
        } catch (ExecutionException | TimeoutException ex) {
            throw new LogBrokerWriterException(ex);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    private CompletableFuture<Integer> writeImpl(List<D> records) {
        logger.debug("{} records to write", records.size());
        return writeImpl(records.iterator());
    }

    /**
     * Схлопывает возрастающие подряд значения в список отрезков.
     */
    static class RangeArray {
        public static final int STRING_SIZE_LIMIT = 1000;
        private Set<Long> values = ConcurrentHashMap.newKeySet();

        /**
         * Количество вставок.
         */
        long getSize() {
            return values.size();
        }

        void addValue(long newVal) {
            values.add(newVal);
        }

        /**
         * Пишет строку в формате [a..b], c, [d..e]. Обрезает если больше 1000 символов.
         */
        @Override
        public String toString() {
            var result = StreamEx.of(values)
                    .sorted()
                    .groupRuns((a, b) -> b == a + 1)
                    .map(l -> l.size() == 1 ? "" + l.get(0) : l.get(0) + "-" + l.get(l.size() - 1))
                    .joining(",");
            if (result.length() > STRING_SIZE_LIMIT) {
                return result.substring(0, STRING_SIZE_LIMIT - 3) + "...";
            } else {
                return result;
            }
        }
    }


    CompletableFuture<Integer> writeImpl(Iterator<D> records) {
        return logbrokerProducerFuture.thenCompose(producer -> {
            Long firstSeqId = null;
            Long lastSeqId = null;
            RangeArray alreadyWrittenRanges = new RangeArray();

            List<CompletableFuture<Void>> writeRequests = new ArrayList<>();
            AtomicInteger writingMessages = new AtomicInteger();
            while (records.hasNext()) {
                LogbrokerWriteRequest request = makeRequest(records.next());
                Long seqId = request.getSeqNo();
                if (firstSeqId == null) {
                    firstSeqId = seqId;
                }
                lastSeqId = seqId;
                CompletableFuture<Void> response = producer.write(request.getData(), seqId, request.getCreateTime()).thenAccept(r -> {
                    if (r.isAlreadyWritten()) {
                        alreadyWrittenRanges.addValue(r.getSeqNo());
                    } else {
                        writingMessages.incrementAndGet();
                    }
                });

                writeRequests.add(response);
            }

            logger.debug("{} records with seqId from {} to {} are going to be written",
                    writeRequests.size(), firstSeqId, lastSeqId);

            return CompletableFuture.allOf(writeRequests.toArray(new CompletableFuture[0]))
                    .thenApply(unused -> {
                        if (alreadyWrittenRanges.getSize() > 0) {
                            logger.warn("{} records were already written. seqIds: {}",
                                    alreadyWrittenRanges.getSize(), alreadyWrittenRanges);
                        }
                        return writingMessages.get();
                    });
        });
    }

    private CompletableFuture<AsyncProducer> restartLogbrokerProducer() {
        logger.debug("Close lock are going to be received by restartLogbrokerProducer method");
        closeLock.lock();
        try {
            if (isClosed) {
                throw new CancellationException(
                        "Logbroker producer closed while trying to retry writing to lb, cancel retry");
            }
            CompletableFuture<Void> closeProducerFuture =
                    logbrokerProducerFuture.thenAccept(AsyncProducer::close).exceptionally(
                            ex -> {
                                logger.error("Got exception on Logbroker session close. Ignoring.", ex);
                                return null;
                            });

            CompletableFuture<AsyncProducer> asyncProducerFuture =
                    closeProducerFuture
                            .thenCompose(unused -> logbrokerProducerSupplier.get());

            logbrokerProducerFuture =
                    asyncProducerFuture
                            .thenCompose(AsyncProducer::init)
                            .thenAccept(this::acceptInitResponse)
                            .thenCompose(unused -> asyncProducerFuture);
        } finally {
            closeLock.unlock();
            logger.debug("Close lock released by restartLogbrokerProducer method");
        }
        return logbrokerProducerFuture;
    }

    private void acceptInitResponse(ProducerInitResponse producerInitResponse) {
        initSeqNo = producerInitResponse.getMaxSeqNo();
        currentSeqNo.set(initSeqNo);
    }

    @Override
    @PreDestroy
    public void close() {
        if (isClosed) {
            return;
        }
        logger.debug("Close lock are going to be received by close method");
        closeLock.lock();
        try {
            if (logbrokerProducerFuture != null) {
                logbrokerProducerFuture.thenAccept(AsyncProducer::close).get(logbrokerTimeoutSec, SECONDS);
            }
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        } catch (ExecutionException | TimeoutException ex) {
            throw new LogbrokerWriterCloseException(ex);
        } finally {
            isClosed = true;
            closeLock.unlock();
            logger.debug("Close lock released by close method");
        }
    }

    @Override
    public Long getInitialMaxSeqNo() {
        return initSeqNo;
    }

    private long getAutoincrementSeqNo() {
        return currentSeqNo.incrementAndGet();
    }

    public final class LogbrokerWriteRequest {
        private final byte[] data;
        private final long seqNo;
        private final long createTime;

        public LogbrokerWriteRequest(byte[] data) {
            this(data, getAutoincrementSeqNo());
        }

        public LogbrokerWriteRequest(byte[] data, long seqNo) {
            this(data, seqNo, System.currentTimeMillis());
        }

        public LogbrokerWriteRequest(byte[] data, long seqNo, long createTime) {
            this.data = data;
            this.seqNo = seqNo;
            this.createTime = createTime;
        }

        private byte[] getData() {
            return data;
        }

        private long getSeqNo() {
            return seqNo;
        }

        private long getCreateTime() {
            return createTime;
        }
    }

    public class LogbrokerWriteRequestBuilder {
        private byte[] data;
        private Long seqNo;
        private Long createTime;

        public LogbrokerWriteRequestBuilder withData(byte[] data) {
            this.data = data;
            return this;
        }

        public LogbrokerWriteRequestBuilder withSeqNo(long seqNo) {
            this.seqNo = seqNo;
            return this;
        }

        public LogbrokerWriteRequestBuilder withCreateTime(long createTime) {
            this.createTime = createTime;
            return this;
        }

        public LogbrokerWriteRequest build() {
            return new LogbrokerWriteRequest(data, nvl(seqNo, AbstractLogbrokerWriterImpl.this::getAutoincrementSeqNo),
                    nvl(createTime, System::currentTimeMillis));
        }
    }
}
