package ru.yandex.direct.binlogbroker.logbroker_utils.writer;

import java.time.Duration;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkArgument;

public class CompletableFutureUtils {
    private static final Logger logger = LoggerFactory.getLogger(CompletableFutureUtils.class);
    private static final Duration DEFAULT_DELAY = Duration.ofMillis(5000);
    private static final Supplier<CompletableFuture<Void>> DO_NOTHING = () -> CompletableFuture.completedFuture(null);


    private CompletableFutureUtils() {
    }

    public static <T, U> CompletableFuture<T> wrapCompletableFutureToRetry(
            @Nonnull Supplier<CompletableFuture<T>> action, Supplier<CompletableFuture<U>> exceptionAction, int times,
            Duration delay) {
        checkArgument(times >= 0, "Retry count must be greater or equal then 0");
        CompletableFuture<T> future = action.get();
        for (int i = 0; i < times; i++) {
            int retryCnt = i;
            future = future.thenApply(CompletableFuture::completedFuture)
                    .exceptionally(ex -> {
                                if (ex.getCause() instanceof CancellationException) {
                                    throw (CancellationException) ex.getCause();
                                }
                                logger.warn("Failed to execute supplier, {} attempts left", times - retryCnt, ex);
                                return exceptionAction.get()
                                        .thenCompose(unused -> timer(delay))
                                        .thenCompose(v -> action.get());
                            }
                    )
                    .thenCompose(Function.identity());
        }
        return future;
    }

    public static <T> CompletableFuture<T> wrapCompletableFutureToRetry(
            @Nonnull Supplier<CompletableFuture<T>> action,
            int times
    ) {
        return wrapCompletableFutureToRetry(action, DO_NOTHING, times, DEFAULT_DELAY);
    }

    public static <T> CompletableFuture<T> wrapCompletableFutureToRetry(
            @Nonnull Supplier<CompletableFuture<T>> action,
            Supplier<CompletableFuture<Void>> exceptionAction, int times
    ) {
        return wrapCompletableFutureToRetry(action, exceptionAction, times, DEFAULT_DELAY);
    }

    public static <T> CompletableFuture<T> wrapCompletableFutureToRetry(@Nonnull Supplier<CompletableFuture<T>> action,
                                                                        int times, Duration delay) {
        return wrapCompletableFutureToRetry(action, DO_NOTHING, times, delay);
    }

    private static class TimerHolder {
        private static final Timer TIMER = new Timer("completableFutureUtilsTimer", true);
    }

    private static CompletableFuture<Void> timer(Duration delay) {
        CompletableFuture<Void> cf = new CompletableFuture<>();
        TimerHolder.TIMER.schedule(new TimerTask() {
            @Override
            public void run() {
                cf.complete(null);
            }
        }, delay.toMillis());
        return cf;
    }
}
