package ru.yandex.direct.binlogbroker.logbroker_utils.writer;

import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;

import static java.nio.charset.StandardCharsets.UTF_8;

public class JsonLinesLogbrokerWriter<D> implements LogbrokerWriter<D> {
    private final BatchLogbrokerWriter<D> batchLogbrokerWriter;
    private final int partitionSize;
    private static final Logger logger = LoggerFactory.getLogger(JsonLinesLogbrokerWriter.class);

    public JsonLinesLogbrokerWriter(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                    Duration logbrokerTimeout,
                                    int retryCount, int partitionSize) {
        this.partitionSize = partitionSize;
        batchLogbrokerWriter = new BatchLogbrokerWriter<>(logbrokerProducerSupplier, logbrokerTimeout,
                retryCount, this::serializer);
    }

    AbstractLogbrokerWriterImpl.LogbrokerWriteRequest serializer(List<D> lst) {
        String jsonRequest = lst.stream().map(JsonUtils::toJson).collect(Collectors.joining("\n"));
        return batchLogbrokerWriter.new LogbrokerWriteRequest(jsonRequest.getBytes(UTF_8));
    }

    @Override
    public int writeSync(List<D> data) {
        return batchLogbrokerWriter.writeSync(Lists.partition(Objects.requireNonNull(data), partitionSize));
    }

    @Override
    public CompletableFuture<Integer> write(List<D> data) {
        return batchLogbrokerWriter.write(Lists.partition(Objects.requireNonNull(data), partitionSize));
    }

    @Override
    public Long getInitialMaxSeqNo() {
        return batchLogbrokerWriter.getInitialMaxSeqNo();
    }

    @Override
    public void close() throws InterruptedRuntimeException {
        batchLogbrokerWriter.close();
    }
}
