package ru.yandex.direct.binlogbroker.logbroker_utils.writer;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.utils.InterruptedRuntimeException;

@ParametersAreNonnullByDefault
public interface LogbrokerWriter<D> extends AutoCloseable {
    /**
     * Обработать пачку {@code D data}
     *
     * @param data данные
     * @return количество реально записанных сообщений в логброкер(если сообщение уже было записанно, оно в это
     * счетчик не попадает)
     * @throws LogBrokerWriterException Бросается при первой неудачной обработки какого-либо события.
     */
    CompletableFuture<Integer> write(List<D> data);

    default int writeSync(List<D> data) {
        CompletableFuture<Integer> completableFuture = write(data);
        try {
            return completableFuture.get();
        } catch (ExecutionException ex) {
            throw new LogBrokerWriterException(ex);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    /**
     * Значение maxSeqNo, полученные от сервера при подключении.
     *
     * @return начальное значение maxSeqNo, сообщенное сервером при подключении, null, если writer еще не проинициализирован
     */
    Long getInitialMaxSeqNo();

    @Override
    void close() throws InterruptedRuntimeException;
}
